'use strict';

import isObject from 'isobject';
import WebSocket from 'ws';
import { correctMetadata, createConnectMessage, getSegment } from './utilities';

export type TokenRefreshFn = (token: string) => void;
export type TokenExpiredFn = (fn: TokenRefreshFn) => boolean;

export type Configuration = {
  token: string;
  broadcasterIds: string[];
  gameId: string;
  environment: string;
  onTokenExpired: TokenExpiredFn;
  initialState: State;
  timeoutMs?: number;
};

export type State = { [key: string]: any };

export interface IMDaaS {
  connect: (configuration: Configuration) => Promise<void>;
  disconnect: () => void;
  replaceState: (state: State) => void;
  removeField: (path: string) => void;
  updateField: (path: string, value: any) => void;
  appendToArrayField: (path: string, values: any[]) => void;
}

enum MessageType {
  Append,
  Reauthorize,
  Reconnect,
  Refresh,
  Remove,
  Update,
}

type QueueMessage = {
  type: MessageType;
  path?: string;
  value?: any;
};

const maximumConnectionSize = 100000;
const maximumDeltaSize = 5000;
const pathRx = /^\w+(\.\w+|\[[0-9]\])*$/;
const url: string = process.env['TWITCH_TEST_METADATA_URL'] || 'wss://metadata.twitch.tv/api/ingest';

export default function createMDaaS(): IMDaaS {
  let currentState: State | undefined;
  let webSocket: WebSocket;
  let token: string;
  let connectFn: (state: State) => Error | undefined;
  let openFn: () => Promise<void>;
  let onTokenExpired: TokenExpiredFn;
  let senderTask: any;
  const queue: QueueMessage[] = [];
  const uniqueMessageTypes: MessageType[] = [MessageType.Reconnect, MessageType.Reauthorize, MessageType.Refresh];

  return {
    connect,
    disconnect,
    replaceState,
    removeField,
    updateField,
    appendToArrayField,
  };

  async function connect(configuration: Configuration): Promise<void> {
    if(currentState) {
      throw new Error('already connected');
    }

    // Validate the configuration.
    if(!isObject(configuration.initialState)) {
      throw new Error('initial state is not an object');
    }
    const state = JSON.parse(JSON.stringify(configuration.initialState)) as State;
    correctMetadata(state);
    onTokenExpired = configuration.onTokenExpired;
    token = configuration.token.valueOf().toString();

    // Compose the connect function.
    const broadcasterIds = configuration.broadcasterIds.map((value) => value.valueOf().toString());
    const gameId = configuration.gameId.valueOf().toString();
    const environment = configuration.environment.valueOf().toString();
    if(createConnectMessage(token, broadcasterIds, gameId, environment, state).length > maximumConnectionSize) {
      throw new Error('initial state is too large');
    }
    connectFn = (state: State): Error | undefined => {
      try {
        // Send the "Connect" message.
        webSocket.send(createConnectMessage(token, broadcasterIds, gameId, environment, state));
      } catch(ex) {
        return ex;
      }
    };

    // Compose the open function.
    const timeoutMs = (configuration.timeoutMs || 9999).valueOf();
    openFn = async (): Promise<void> => {
      // Create the socket and await a connection.
      webSocket = await new Promise<WebSocket>((resolve, reject) => {
        const webSocket = new WebSocket(url);
        webSocket.addEventListener('error', onError);
        webSocket.addEventListener('open', onOpen);
        const timerId = setTimeout(onTimeout, timeoutMs);
        function onError(event: WebSocket.ErrorEvent): void {
          clearHandlers();
          reject(event.error);
        }
        function onOpen(_event: WebSocket.OpenEvent): void {
          clearHandlers();
          resolve(webSocket);
        }
        function onTimeout() {
          clearHandlers();
          reject(new Error('timeout expired'));
        }
        function clearHandlers() {
          webSocket.removeEventListener('error', onError);
          webSocket.removeEventListener('open', onOpen);
          clearTimeout(timerId);
        }
      });
      webSocket.addEventListener('close', onClose);
      webSocket.addEventListener('message', onMessage);
    };

    // Open the WebSocket.
    await openFn();

    // Enqueue a "Reauthorize" message without clearing the token.
    queue.push({ type: MessageType.Reauthorize, value: true });

    // Set the current state.
    currentState = state;

    // Invoke sendMessage to send the intial connect message.  The "Reauthorize"
    // message handler will start the sender task.
    await sendMessage();
  }

  function disconnect() {
    clearTimeout(senderTask);
    senderTask = undefined;
    closeWebSocket();
    currentState = undefined;
  }

  function replaceState(state: State): void {
    // Ensure connect has already been invoked.
    if(!currentState) {
      throw new Error('connection not established');
    }

    // Validate the state.
    if(!isObject(state)) {
      throw new Error('state is not an object');
    }
    state = JSON.parse(JSON.stringify(state));

    // Merge required metadata into the state, if necessary.
    const metadata = currentState['_metadata'];
    correctMetadata(state, metadata['id'], metadata['active']);

    // Update the current state.
    currentState = state;

    // Enqueue the "Refresh" message.
    queue.push({ type: MessageType.Refresh });
  }

  function removeField(path: string): void {
    // Ensure connect has already been invoked.
    if(!currentState) {
      throw new Error('connection not established');
    }

    // Validate the path.
    path = path.valueOf();
    if(!path) {
      throw new Error('path is empty');
    } else if(typeof path !== 'string') {
      throw new Error('path is not a string');
    } else if(path.endsWith(']')) {
      throw new Error(`"${path}" does not specify a field`);
    } else if(path === '_metadata' || path === '_metadata.active' || path === '_metadata.id') {
      throw new Error(`cannot remove "${path}"`);
    } else if(!pathRx.test(path)) {
      throw new Error(`"${path}" is not a valid field specifier`);
    }
    const segment = getSegment(currentState, path);
    if(!segment || typeof segment.parent[segment.field] === 'undefined') {
      throw new Error(`"${path}" does not specify a known field`);
    }

    // Update the current state.
    const parent = segment.parent as State;
    const field = segment.field as string;
    delete parent[field];

    // Enqueue the "Remove Field" message.
    queue.push({ type: MessageType.Remove, path });
  }

  function updateField(path: string, value: any): void {
    // Ensure connect has already been invoked.
    if(!currentState) {
      throw new Error('connection not established');
    }

    // Validate the path and value.
    path = path.valueOf();
    value = JSON.parse(JSON.stringify(value));
    if(!path) {
      throw new Error('path is empty');
    } else if(typeof path !== 'string') {
      throw new Error('path is not a string');
    } else if(path === '_metadata' && !isObject(value)) {
      throw new Error('_metadata value is not an object');
    } else if(path === '_metadata.active' && typeof value !== 'boolean') {
      throw new Error('_metadata.active value is not Boolean');
    } else if(path === '_metadata.id' && typeof value !== 'string') {
      throw new Error('_metadata.id value is not a string');
    } else if(!pathRx.test(path)) {
      throw new Error(`"${path}" is not a valid field specifier`);
    }
    const segment = getSegment(currentState, path);
    if(!segment) {
      throw new Error(`"${path}" does not specify a known field`);
    } else if(Array.isArray(segment.parent) && segment.field >= segment.parent.length) {
      throw new Error(`"${path}" is out of bounds`);
    }

    // Merge required metadata into the value, if necessary.
    if(path === '_metadata') {
      const metadataState = { '_metadata': value };
      const metadata = currentState['_metadata'];
      correctMetadata(metadataState, metadata['id'], metadata['active']);
      value = metadataState['_metadata'];
    }

    // Update the current state.
    segment.parent[segment.field] = value;

    // Enqueue the "Update" message.
    queue.push({ type: MessageType.Update, path, value });
  }

  function appendToArrayField(path: string, values: any[]): void {
    // Ensure connect has already been invoked.
    if(!currentState) {
      throw new Error('connection not established');
    }

    // Validate the path and values.
    path = path.valueOf();
    values = JSON.parse(JSON.stringify(values));
    if(!path) {
      throw new Error('path is empty');
    } else if(typeof path !== 'string') {
      throw new Error('path is not a string');
    } else if(!pathRx.test(path)) {
      throw new Error(`"${path}" is not a valid field specifier`);
    } else if(!Array.isArray(values)) {
      throw new Error('values is not an array');
    }
    const segment = getSegment(currentState, path);
    if(!segment || typeof segment.parent[segment.field] === 'undefined') {
      throw new Error(`"${path}" does not specify a known field`);
    }
    const array = segment.parent[segment.field];
    if(!Array.isArray(array)) {
      throw new Error(`"${path}" does not specify an array field`);
    } else if(values.length === 0) {
      console.warn('[MDaaS.appendToArrayField] array is empty; ignoring');
      return;
    }

    // Update the current state.
    array.push(...values);

    // Enqueue the "Append" message.
    queue.push({ type: MessageType.Append, path, value: values });
  }

  async function acquireToken(): Promise<string> {
    return await new Promise<string>((resolve, _reject) => {
      try {
        if(!onTokenExpired(resolve)) {
          // The client wants to shut down.
          resolve('');
        }
      } catch(ex) {
        // There is likely a logic error in the client.  Shut down the connection.
        console.error('[MDaaS.acquireToken] onTokenExpired threw an exception:', ex);
        resolve('');
      }
    });
  }

  async function sendMessage(): Promise<void> {
    try {
      senderTask = undefined;
      const uniqueMessage = ([] as QueueMessage[]).concat(...uniqueMessageTypes.map((t) => queue.find((m) => m.type === t) || []))[0];
      if(uniqueMessage) {
        // There is a unique message.  Clear the queue and process the message.
        queue.splice(0, queue.length);
        switch(uniqueMessage.type) {
          case MessageType.Reauthorize:
            if(!uniqueMessage.value) {
              token = '';
            }
            await reauthorize();
            break;
          case MessageType.Reconnect:
            // Await the requested delay.
            const delay = uniqueMessage.value - Date.now();
            if(delay > 0) {
              await new Promise((resolve, _reject) => setTimeout(resolve, delay));
            }

            if(currentState) {
              // Open a new connection.
              await openFn();

              // Perform the "Reauthorize" action above.
              if(currentState) {
                await reauthorize();
              }
            }
            break;
          case MessageType.Refresh:
            webSocket.send(JSON.stringify({ refresh: { data: currentState } }));
            break;
          default:
            throw new Error(`unexpected message type ${uniqueMessage.type}`);
        }
      } else {
        // Trim the queue based on repeated modifications.
        const trimmedQueue: QueueMessage[] = [];
        while(queue.length) {
          const message = queue.splice(0, 1)[0];
          switch(message.type) {
            case MessageType.Append:
              const currentMessage = trimmedQueue.find((m) => m.type === message.type && m.path === message.path);
              if(currentMessage) {
                currentMessage.value.push(...message.value);
              } else {
                trimmedQueue.push(message);
              }
              break;
            case MessageType.Remove:
            case MessageType.Update:
              trimmedQueue.splice(0, trimmedQueue.length, ...trimmedQueue.filter((m) => m.path !== message.path));
              trimmedQueue.push(message);
              break;
            default:
              throw new Error(`unexpected message type ${message.type}`);
          }
        }
        queue.splice(0, queue.length, ...trimmedQueue);

        // Take messages off of the queue until reaching the maximum payload size.
        if(queue.length) {
          const deltas: any[] = [];
          while(queue.length) {
            let delta: any;
            const message = queue[0];
            switch(message.type) {
              case MessageType.Append:
                delta = [message.path, 'a', message.value];
                break;
              case MessageType.Remove:
                delta = [message.path];
                break;
              case MessageType.Update:
                delta = [message.path, message.value];
                break;
              default:
                throw new Error(`unexpected message type ${message.type}`);
            }
            if(JSON.stringify({ delta: [...deltas, delta] }).length > maximumDeltaSize) {
              break;
            }
            deltas.push(delta);
            queue.shift();
          }
          if(deltas.length) {
            webSocket.send(JSON.stringify({ delta: deltas }));
          } else {
            // The first message is larger than the maximum payload size.  Enqueue a
            // "Refresh" message and recurse.
            queue.splice(0, queue.length, { type: MessageType.Refresh });
            await sendMessage();

            // Prevent two time-outs due to recursion.
            clearTimeout(senderTask);
          }
        }
      }
    } catch(ex) {
      console.error('[MDaaS.sendMessage]', ex.message);
    } finally {
      // Set another time-out to send the next message.
      if(currentState) {
        senderTask = setTimeout(sendMessage, 1000);
      }
    }
  }

  async function reauthorize(): Promise<boolean> {
    if(!token) {
      token = await acquireToken();
    }
    if(token) {
      const error = connectFn(currentState!);
      if(!error) {
        return true;
      }
      console.error('[MDaaS.reauthorize] connection error', error.message);
    }
    disconnect();
    return false;
  }

  function closeWebSocket() {
    webSocket.removeEventListener('close', onClose);
    webSocket.removeEventListener('message', onMessage);
    webSocket.close();
  }

  function onClose(_event: WebSocket.CloseEvent): void {
    // Clear the handlers and enqueue a "Reconnect" message.
    webSocket.removeEventListener('close', onClose);
    webSocket.removeEventListener('message', onMessage);
    queue.push({ type: MessageType.Reconnect, value: Date.now() });
  }

  function onMessage(event: WebSocket.MessageEvent): void {
    try {
      const data = event.data.toString();
      const response = JSON.parse(data);
      const keys = Object.keys(response);
      if(keys.length === 1) {
        switch(keys[0]) {
          case 'connected':
            if(response['connected']) {
              return;
            }
            break;
          case 'error':
            const error = response['error'];
            if(isObject(error)) {
              const code = error['code'];
              if(code === 'connect_invalid_token') {
                // The server has rejected the authorization token.
                queue.push({ type: MessageType.Reauthorize });
                return;
              } else if(code === 'connection_not_authed') {
                // Ignore this since it means we're in the middle of getting connected.
                return;
              } else if(code === 'waiting_on_refresh_message') {
                // Ignore this since it means we're in the middle of refreshing.
                return;
              }
            }
            break;
          case 'reconnect':
            const reconnectDelay = response['reconnect'];
            if(typeof reconnectDelay === 'number') {
              // The server requested a reconnection.  Close the socket.  The reconnection
              // process will open a new one.
              const reconnectTime = reconnectDelay + Date.now();
              closeWebSocket();
              queue.push({ type: MessageType.Reconnect, value: reconnectTime });
              return;
            }
            break;
        }
      }
      throw new Error(`unexpected response from server:  ${data}`);
    } catch(ex) {
      console.error('[MDaaS.onMessage]', ex.message);
    }
  }
}
