'use strict';

import isObject from 'isobject';
import { State } from '.';

export function createConnectMessage(token: string, broadcasterIds: string[], gameId: string, environment: string, state: State) {
  return JSON.stringify({
    connect: {
      token,
      broadcaster_ids: broadcasterIds,
      game_id: gameId,
      env: environment,
      data: state,
    },
  });
}

export function getSegment(state: State, path: string): { parent: State, field: string | number } | undefined {
  const parts = path.split(/[.[]/);
  const field = parts.pop();
  if(field) {
    const parent = parts.reduce(checkSegment, state);
    if(parent) {
      if(field.endsWith(']')) {
        if(Array.isArray(parent)) {
          const index = parseInt(field, 10);
          return { parent, field: index };
        }
      } else if(isObject(parent)) {
        return { parent, field };
      }
    }
  }

  function checkSegment(state: State | undefined, segment: string): State | undefined {
    if(state) {
      if(segment.endsWith(']')) {
        if(Array.isArray(state)) {
          const index = parseInt(segment, 10);
          return state[index];
        }
      } else if(isObject(state)) {
        return state[segment];
      }
    }
  }
}

const idCharacters: string = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';

export function correctMetadata(state: State, id?: string, isActive?: boolean): void {
  // Add the "_metadata" field if not present.  If it is present, ensure it
  // contains the required fields of the expected types.
  const metadata = state['_metadata'];
  if(metadata === undefined) {
    state['_metadata'] = {
      id: id || generateSessionId(),
      active: isActive === undefined ? true : isActive,
    };
  } else {
    if(!isObject(metadata)) {
      throw new Error('_metadata value is not an object');
    }
    if(metadata['id'] === undefined) {
      metadata['id'] = id || generateSessionId();
    } else if(typeof metadata['id'] !== 'string') {
      throw new Error('_metadata.id value is not a string');
    }
    if(metadata['active'] === undefined) {
      metadata['active'] = true;
    } else if(typeof metadata['active'] !== 'boolean') {
      throw new Error('_metadata.active value is not a Boolean');
    }
  }

  function generateSessionId() {
    return Array(32).fill(0).map(_ => idCharacters.charAt(Math.floor(idCharacters.length * Math.random()))).join('');
  }
}
