package authentication

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/devhub/gdaas-ingest/models"
	"code.justin.tv/foundation/twitchclient"
)

// When feature grows larger, we should move whitelisting to more automatic ways
var whitelistedClient = map[string]int{
	"ua4ag5pg919urjgkc9nzqv1z77n9jc":  1234, // Shiming's Test App to fake game id
	"5eyrj8lcpf6cb3wye5k4zu9u85b69zd": 456,  // Another Shiming's Test App
}

//go:generate mockery -name Auth
type Auth interface {
	ValidateClient(ctx context.Context, token string) (string, error)
}

type authImpl struct {
	twitchclient.JSONClient
}

func NewClient(host string, conf twitchclient.ClientConf) *authImpl {
	httpClient := twitchclient.NewHTTPClient(conf)
	jsonClient := twitchclient.NewJSONClient(host, httpClient)
	return &authImpl{*jsonClient}
}

func (a *authImpl) ValidateClient(ctx context.Context, token string) (string, error) {
	clientID, err := a.getClientInfo(ctx, token)
	if err != nil {
		return "", err
	}

	if err := a.ifWhitelistedClient(clientID); err != nil {
		return "", err
	}

	return clientID, nil
}

func (a *authImpl) getClientInfo(ctx context.Context, token string) (string, error) {
	headers := []string{fmt.Sprintf("OAuth %s", token)}
	jsonRequest := twitchclient.JSONRequest{
		Method: "GET",
		Path:   twitchclient.Pathf("/oauth2/validate"),
		Query:  url.Values{},
		Header: http.Header{
			"Authorization": headers,
		},
	}
	var authResult models.AuthResp
	resp, err := a.Do(ctx, jsonRequest, &authResult)
	if err != nil || resp.StatusCode >= 400 {
		return "", err
	}
	return authResult.ClientID, err
}

func (a *authImpl) ifWhitelistedClient(clientID string) error {
	if _, clientFound := whitelistedClient[clientID]; clientFound {
		return nil
	}
	return errors.New("client not whitelisted")
}
