package harness

import (
	"errors"
	"net"
	"sync"

	"code.justin.tv/devhub/gdaas-ingest/libs/session"
)

var ErrNotConnected = errors.New("Not connected")

type MatchFunction func([]byte, bool) bool

type Driver interface {
	session.Client
	session.Binding
	WaitFor(match MatchFunction) (data []byte, isText bool, err error) // note: blocks until reciept or close
	Read() (msg []byte, isText bool, err error)                        // next message no matter its content
	IsClosed() bool
	attach(session.Link)
}

type msg struct {
	isText bool
	bytes  []byte
}

type mockDriver struct {
	addr   net.Addr
	closed bool
	mutex  sync.RWMutex
	input  *sync.Cond
	link   session.Link
	msgs   []*msg
}

func NewDriver(id string) Driver {
	m := &mockDriver{addr: &mockAddr{id}}
	m.input = sync.NewCond(&m.mutex)
	return m
}

// implement session.Binding
func (m *mockDriver) OnTextMessage(str string) {
	m.mutex.Lock()
	m.msgs = append(m.msgs, &msg{isText: true, bytes: []byte(str)})
	m.input.Signal()
	m.mutex.Unlock()
}

func (m *mockDriver) OnBinaryMessage(bytes []byte) {
	m.mutex.Lock()
	m.msgs = append(m.msgs, &msg{bytes: bytes})
	m.input.Signal()
	m.mutex.Unlock()
}

func (m *mockDriver) OnClosed(err error) {
	m.mutex.Lock()
	m.link = nil
	m.closed = true
	m.input.Signal()
	m.mutex.Unlock()
}

// implement session.Link
func (m *mockDriver) WriteText(str string) error {
	m.mutex.RLock()
	link := m.link
	m.mutex.RUnlock()
	if link == nil {
		return ErrNotConnected
	}
	return link.WriteText(str)
}

func (m *mockDriver) WriteBinary(bytes []byte) error {
	m.mutex.RLock()
	link := m.link
	m.mutex.RUnlock()
	if link == nil {
		return ErrNotConnected
	}
	return link.WriteBinary(bytes)
}

func (m *mockDriver) Close(err error) {
	m.mutex.Lock()
	m.closed = true
	link := m.link
	m.link = nil
	m.input.Broadcast()
	m.mutex.Unlock()
	if link != nil {
		link.Close(err)
	}
}

func (m *mockDriver) WriteBinaryAsText(bytes []byte) error {
	m.mutex.RLock()
	link := m.link
	m.mutex.RUnlock()
	if link == nil {
		return ErrNotConnected
	}
	return link.WriteBinaryAsText(bytes)
}

// implement Client add-ons to session.Binding
func (m *mockDriver) Address() net.Addr { return m.addr }
func (m *mockDriver) attach(srv session.Link) {
	m.mutex.Lock()
	m.link = srv
	m.mutex.Unlock()
}

// implement additional functions
func (m *mockDriver) IsClosed() bool {
	m.mutex.RLock()
	defer m.mutex.RUnlock()
	return m.closed
}

func (m *mockDriver) WaitFor(match MatchFunction) (data []byte, isText bool, err error) {
	data = nil
	err = nil
	isText = false
	m.mutex.Lock()
	defer m.mutex.Unlock()
	for {
		for i, msg := range m.msgs {
			if match(msg.bytes, msg.isText) {
				isText = msg.isText
				data = msg.bytes
				m.msgs = append(m.msgs[:i], m.msgs[i+1:]...)
				return
			}
		}
		if m.closed {
			err = ErrNotConnected
			return
		}
		m.input.Wait() // hang out until close or message received
	}
}

func (m *mockDriver) Read() (msg []byte, isText bool, err error) {
	return m.WaitFor(func(msg []byte, isText bool) bool { return true })
}
