package websocket

import (
	"context"
	"net/url"
	"sync/atomic"
	"unsafe"

	"code.justin.tv/devhub/gdaas-ingest/libs/session"
	"code.justin.tv/extensions/shutdown"
	"github.com/gobwas/ws"
)

func ClientFactory(url *url.URL, settings *Settings) session.ClientFactory {
	return func(bind session.Binding) (session.Client, error) {
		return NewClient(url, bind, settings)
	}
}

func NewClientResolver(settings *Settings) session.ClientResolver {
	return func(url *url.URL, bind session.Binding) (session.Client, error) {
		return NewClient(url, bind, settings)
	}
}

func NewClient(url *url.URL, bind session.Binding, settings *Settings) (session.Client, error) {
	conn, _, _, err := ws.Dialer{TLSConfig: settings.Certs}.Dial(context.Background(), url.String())
	if err != nil {
		return nil, err
	}

	internal := fixup(settings)
	internal.Lifecycle = shutdown.NewManager()

	// shut down the client lifecycle to close; in practice this makes the client
	// send an appropriate close message to the service which causes graceful
	// termination.
	link := &link{conn: conn, mask: ws.MaskFrameInPlace, logger: settings.Logger, onClose: func(error) { go internal.Lifecycle.Shutdown() }}

	if settings.Lifecycle != nil {
		// if we're in a managed lifecycle, then remove ourselves as a managed resource on close
		// (this is still a call to shutdown internal under the hood)
		settings.Lifecycle.RegisterHook(internal, func() error {
			for _, err := range internal.Lifecycle.Shutdown() {
				return err
			}
			return nil
		})

		// if we're in a managed lifecycle, then remove ourselves as a managed resource on close	 		link.onClose = func(error) {
		// (this is still a call to shutdown safeSettings under the hood)
		link.onClose = func(error) {
			go settings.Lifecycle.ExecuteHook(internal)
		}
	}

	// request client connection close on client shutdown request
	closed := false
	result := unsafe.Pointer(nil)
	internal.Lifecycle.RegisterHook(bind, func() error {
		var out error
		if pErr := (*error)(atomic.LoadPointer(&result)); pErr != nil {
			out = *pErr
		}
		bind.OnClosed(out)
		return nil
	})

	internal.Lifecycle.RegisterHook(link, func() error {
		if !closed {
			link.writeClose(ws.StatusNormalClosure, "")
		}
		return nil
	})

	internal.Lifecycle.RunUntilComplete(func() {
		defer func() {
			closed = true
			link.Close(err)
		}()
		defer settings.gracefulClose(link.conn)
		value := execute(bind, link, ws.StateClientSide, internal, func() bool { return false })
		atomic.StorePointer(&result, unsafe.Pointer(&value))
	})

	return link, nil
}
