package main

import (
	"crypto/tls"
	"flag"
	"log"
	"os"
	"strconv"
	"time"

	"code.justin.tv/devhub/gdaas-ingest/libs/logging"
	"code.justin.tv/devhub/gdaas-ingest/libs/timeout"
	"code.justin.tv/devhub/gdaas-ingest/libs/websocket"
	"code.justin.tv/devhub/gdaas-ingest/server"
	"code.justin.tv/extensions/shutdown"
)

const (
	defaultPort = 3004
)

func checkEnv(key, def string) string {
	if value, ok := os.LookupEnv(key); ok {
		return value
	}
	return def
}

func checkPort() int64 {
	if value, ok := os.LookupEnv("PORT"); ok {
		port, err := strconv.ParseInt(value, 10, 16)
		if err != nil {
			panic("Illegal port: " + value)
		}
		return port
	}
	return defaultPort
}

func main() {
	port := flag.Int64("port", checkPort(), "port for incoming connections")
	logLevel := flag.String("log", checkEnv("LOG", "info"), "set logging level: [trace,debug,info,warning,error]")
	certFile := flag.String("cert-file", checkEnv("CERT_FILE", ""), "file containing TLS server cert")
	keyFile := flag.String("key-file", checkEnv("KEY_FILE", ""), "file containing TLS server key")
	flag.Parse()

	mgr := shutdown.NewManager()
	logger := logging.NewFilter(logging.ParseLevel(*logLevel), log.Println).Log
	defer exit(mgr, logger)

	tls, err := loadTLS(*certFile, *keyFile)
	if err != nil {
		logger(logging.Error, "Unable to load TLS", err)
		return
	}

	settings := &websocket.Settings{
		Certs:     tls,
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewPrecomputedSampler(100, 10*time.Second, 1*time.Second),
	}

	websocket.NewService(int(*port), server.Factory, settings).Start()
	logger(logging.Info, "Exiting on signal:", mgr.ListenForInterrupt())
}

func exit(mgr shutdown.Manager, logger logging.Function) {
	errs := mgr.Shutdown()
	for _, err := range errs {
		logger(logging.Error, "Shutdown error:", err)
	}
	os.Exit(len(errs))
}

func loadTLS(certFile, keyFile string) (*tls.Config, error) {
	if certFile != "" || keyFile != "" {
		cer, err := tls.LoadX509KeyPair(certFile, keyFile)
		if err != nil {
			return nil, err
		}
		return &tls.Config{Certificates: []tls.Certificate{cer}}, nil
	}
	return nil, nil
}
