package server

import (
	"code.justin.tv/devhub/gdaas-ingest/internal/clients/authentication"
	"code.justin.tv/devhub/gdaas-ingest/libs/session"
	"code.justin.tv/devhub/gdaas-ingest/models"
	"code.justin.tv/foundation/twitchclient"
	"context"
	"encoding/json"
	"errors"
	"github.com/davecgh/go-spew/spew"
	"sync"
	"time"
)

type server struct{}

func (*server) Factory() session.BindingFactory { return Factory }

func Server() session.Server { return &server{} }

// Factory implements session.Factory for a simple echo service
func Factory(client session.Client) session.Binding {
	authClient := authentication.NewClient("https://id.twitch.tv", twitchclient.ClientConf{})
	return &binding{client, authClient, nil, nil, time.Now(), time.Minute, sync.Mutex{}}
}

// binding implements session.Binding for a simple echo service
type binding struct {
	client                 session.Client
	authClient             authentication.Auth
	clientID               *string
	fullStateData          *models.FullStateDataPayload
	lastUpdatedAt          time.Time
	maxAllowedDataInterval time.Duration
	// In order to keep sequenced processing, lock the session binding when process one data pack
	locker sync.Mutex
}

// Simple rate limiter, close connection if more than one message per sec
func (b *binding) RateLimitChecker() {
	if b.fullStateData != nil {
		oneSecBracket := time.Now().Truncate(time.Second)
		if oneSecBracket.Before(b.lastUpdatedAt) {
			b.client.Close(errors.New("reach allowed rate limit"))
		}
	}
}

// Check if the connection is idle for too long time, close it to save resource consuming
func (b *binding) CheckIfIdle() {
	for {
		time.Sleep(b.maxAllowedDataInterval)
		now := time.Now()
		allowedLatestUpdateTime := now.Truncate(b.maxAllowedDataInterval)
		if allowedLatestUpdateTime.After(b.lastUpdatedAt) {
			spew.Dump("client idle for long time\n")
			b.client.Close(errors.New("client idle for long time"))
		}
	}
}

func (b *binding) OnTextMessage(msg string) { b.client.WriteText(msg) }

func (b *binding) OnBinaryMessage(msg []byte) {
	b.locker.Lock()
	var genericData models.GenericDataPack
	if err := json.Unmarshal(msg, &genericData); err != nil {
		b.client.Close(err)
	}

	switch genericData.PackType {
	case models.Connect:
		var connectData models.ConnectDataPayload
		if err := json.Unmarshal(genericData.Payload, &connectData); err != nil {
			b.client.Close(err)
		}
		if err := b.authorizeConnection(connectData.Token); err != nil {
			b.client.Close(err)
		}
		b.fullStateData = &connectData.FullStateData
	case models.Disconnect:
		b.client.Close(nil)
	case models.DeltaData:
		if b.clientID == nil || b.fullStateData == nil {
			b.client.Close(errors.New("connection error: please reconnect"))
		}
		var deltaData models.DeltaDataPayload
		if err := json.Unmarshal(genericData.Payload, &deltaData); err != nil {
			b.client.Close(err)
		}

		if err := b.aggregateData(deltaData); err != nil {
			b.client.Close(err)
		}
	default:
		b.client.Close(errors.New("please specify valid data pack type"))
	}

	b.lastUpdatedAt = time.Now()
	b.client.WriteBinary([]byte("succeed"))
	b.locker.Unlock()
}

func (b *binding) OnClosed(error) {} // no cleanup necessary

func (b *binding) authorizeConnection(token string) error {
	clientID, err := b.authClient.ValidateClient(context.Background(), token)
	if err != nil {
		return err
	}
	b.clientID = &clientID
	return nil
}

func (b *binding) aggregateData(deltaData models.DeltaDataPayload) error {
	return nil
}
