package shutdown

import (
	"os"
	"os/signal"
	"sort"
	"sync"
	"syscall"
)

var nilReporter = func(interface{}, error) {}

// Manager allows registration of shutdown hooks that must be completed before
// a program is allowed to exit.
type Manager interface {
	SetOnError(Reporter)

	RegisterHook(interface{}, Hook) // inserts hook to be executed (stack ordering)
	ExecuteHook(interface{}) error  // runs immediately rather than at shutdown
	RunUntilComplete(func())        // allows launch of goroutines that must complete before shutdown
	ListenForInterrupt() os.Signal  // block for an OS signal that says we need to stop execution
	Shutdown() []error
}

type manager struct {
	sigs    chan os.Signal
	onError Reporter
	hooks   map[interface{}]*entry
	added   int
	mutex   sync.Mutex
	wg      sync.WaitGroup
}

// NewManager creates a new manager
func NewManager() Manager {
	return &manager{
		onError: nilReporter,
		hooks:   make(map[interface{}]*entry),
	}
}

func (m *manager) SetOnError(r Reporter) { m.onError = r }
func (m *manager) RegisterHook(key interface{}, h Hook) {
	m.mutex.Lock()
	m.hooks[key] = &entry{key, h, m.added}
	m.added += 1
	m.mutex.Unlock()
}

func (m *manager) ExecuteHook(key interface{}) error {
	m.mutex.Lock()
	entry, ok := m.hooks[key]
	delete(m.hooks, key)
	m.mutex.Unlock()
	if ok {
		return execute(key, entry.hook, m.onError)
	}
	return nil
}

func (m *manager) ListenForInterrupt() os.Signal {
	sigChan := make(chan os.Signal)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM, syscall.SIGUSR2)
	sig := <-sigChan
	signal.Stop(sigChan)
	return sig
}

func (m *manager) Shutdown() []error {
	errors := m.executeAll()
	m.wg.Wait()
	return errors
}

func (m *manager) RunUntilComplete(async func()) {
	m.wg.Add(1)
	go func() {
		defer m.wg.Done()
		async()
	}()
}

func (m *manager) executeAll() []error {
	errors := []error{}
	m.mutex.Lock()
	listing := make(entries, len(m.hooks))
	i := 0
	for key := range m.hooks {
		listing[i] = m.hooks[key]
		i = i + 1
	}
	m.hooks = map[interface{}]*entry{}
	m.mutex.Unlock()
	sort.Sort(listing)
	for _, entry := range listing {
		if err := execute(entry.key, entry.hook, m.onError); err != nil {
			errors = append(errors, err)
		}
	}
	return errors
}

func execute(key interface{}, hook Hook, reporter Reporter) error {
	if err := hook(); err != nil {
		reporter(key, err)
		return err
	}
	return nil
}
