# JSON-delta

This Go library is used for patching a JSON state object from deltas to create a new JSON state object.

## Data Packets

The data packet is the string representation of an array of deltas to apply to the state. These mutate the state by either replacing the whole object or applying changes to specific fields or array elements.

The array can contain any number of operations and they are applied in order.

Full replacement:
```json
[{ "key": "value", "anotherKey": [1, 2, 3] }]
```

Add or replace a field:
```json
["fieldName", "value"]
```

Remove a field:
```json
["fieldName"]
```

Append a value to an array:
```json
["fieldName", "a", [42]]
```

Append multiple values to an array:
```json
["fieldName", "a", [3, "four", null]]
```

## Examples

Here is an example of using this library. To run it, please see [the example](main.go).

```
func main() {
	deltaStr := `[["test", {"a": {"b": 1}}], ["damage_dealt", 10000], ["champion.skill.e", 2]]`
	fullStateStr := `{
		"damage_dealt": 12092,
		"position": [
			[1, 2],
			[3, 4]
		],
		"champion": {
			"name": "lulu",
			"resource": [1, 2, 3, 4],
			"skill": {
				"q": 1,
				"w": 3
			}
		}
	}`

	fullState, _ := jdelta.FullStateStringToMap(fullStateStr)
	deltas, _ := jdelta.DeltaStateStringToInterface(deltaStr)
	f, err := jdelta.PatchDeltas(fullState, deltas)
	spew.Dump(err)
	spew.Dump(f)
}
```

The output is the updated `fullState` object after applying the deltas given in `deltaStr`.

```
GOROOT=/usr/local/Cellar/go/1.12.5/libexec #gosetup
GOPATH=/Users/shimingr/go #gosetup
/usr/local/Cellar/go/1.12.5/libexec/bin/go build -o /private/var/folders/qp/7kw6x3yn3cx27p4gq97q2fd4yrybhv/T/___go_build_main_go /Users/shimingr/go/src/code.justin.tv/devhub/json-delta/main.go #gosetup
/private/var/folders/qp/7kw6x3yn3cx27p4gq97q2fd4yrybhv/T/___go_build_main_go #gosetup
(interface {}) <nil>
(map[string]interface {}) (len=4) {
 (string) (len=4) "test": (map[string]interface {}) (len=1) {
  (string) (len=1) "a": (map[string]interface {}) (len=1) {
   (string) (len=1) "b": (float64) 1
  }
 },
 (string) (len=12) "damage_dealt": (float64) 10000,
 (string) (len=8) "position": ([]interface {}) (len=2 cap=2) {
  ([]interface {}) (len=2 cap=2) {
   (float64) 1,
   (float64) 2
  },
  ([]interface {}) (len=2 cap=2) {
   (float64) 3,
   (float64) 4
  }
 },
 (string) (len=8) "champion": (map[string]interface {}) (len=3) {
  (string) (len=4) "name": (string) (len=4) "lulu",
  (string) (len=8) "resource": ([]interface {}) (len=4 cap=4) {
   (float64) 1,
   (float64) 2,
   (float64) 3,
   (float64) 4
  },
  (string) (len=5) "skill": (map[string]interface {}) (len=3) {
   (string) (len=1) "e": (float64) 2,
   (string) (len=1) "q": (float64) 1,
   (string) (len=1) "w": (float64) 3
  }
 }
}

Process finished with exit code 0
```

## Best Practices
- New fields can be added to a state object. If the value is a complex object, the correct way of writing the "add" delta is `[["test", {"a": {"b": 1}}]]` instead of `[["test.a.b", 1]]` since MDaaS will not create intermediate objects.
- MDaaS rejects deltas that specify an array index that is out of bounds. If new items needed to be added to an existing array, use an "append" delta.
- The array as a concept is an interface array without type enforcement so different types of values can be appended or replaced. For example, [true,2,null,"4"] is an allowed value to use in an "append" delta.
- The "append" delta applies only to arrays, not maps. If a new value is needed for a map, use an "add" delta.
