package jdelta

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestDeltaToFullState(t *testing.T) {
	t.Run("should be able to deal with remove an item from a map", func(t *testing.T) {
		deltaStr := `["champion.name"]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4]
				],
				"champion": {
					"resource": [1, 2, 3, 4],
					"skill": {
						"q": 1,
						"w": 3
					}
				}
			}`

		fullStateExpected, err := FullStateStringToMap(fullStateExpectedStr)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, fullState)
	})

	t.Run("should be able to deal with replace an item from a map", func(t *testing.T) {
		deltaStr := `["champion.name", "maokai"]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4]
				],
				"champion": {
					"resource": [1, 2, 3, 4],
					"skill": {
						"q": 1,
						"w": 3
					},
					"name": "maokai"
				}
			}`

		fullStateExpected, err := FullStateStringToMap(fullStateExpectedStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpectedStr)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, fullState)
	})

	t.Run("should be able to deal with add an item to a map", func(t *testing.T) {
		deltaStr := `["test", {"a": {"b": 1}}]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4]
				],
				"champion": {
					"name": "lulu",
					"resource": [1, 2, 3, 4],
					"skill": {
						"q": 1,
						"w": 3
					}
				},
				"test": {"a": {"b": 1}}
			}`

		fullStateExpected, err := FullStateStringToMap(fullStateExpectedStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpectedStr)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, fullState)
	})

	t.Run("should be able to deal with add an item to a map within existing scope", func(t *testing.T) {
		deltaStr := `["champion.skill.e", 2]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4]
				],
				"champion": {
					"name": "lulu",
					"resource": [1, 2, 3, 4],
					"skill": {
						"q": 1,
						"w": 3,
						"e": 2
					}
				}
			}`

		fullStateExpected, err := FullStateStringToMap(fullStateExpectedStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpectedStr)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, fullState)
	})

	t.Run("should be able to deal with append item to an array", func(t *testing.T) {
		deltaStr := `["champion.resource", "a", [5]]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4]
				],
				"champion": {
					"name": "lulu",
					"resource": [1, 2, 3, 4, 5],
					"skill": {
						"q": 1,
						"w": 3
					}
				}
			}`

		fullStateExpected, err := FullStateStringToMap(fullStateExpectedStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpectedStr)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, fullState)
	})

	t.Run("should be able to deal with add an item to a map", func(t *testing.T) {
		deltaStr := `["position", "a", [[5,6]]]`

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullState)

		deltas, err := DeltaStateStringTranslateHelper(deltaStr)
		assert.NoError(t, err)

		err = deltaToFullState(fullState, deltas)
		assert.NoError(t, err)

		fullStateExpectedStr :=
			`{
				"damage_dealt": 12092,
				"position": [
					[1, 2],
					[3, 4],
					[5, 6]
				],
				"champion": {
					"name": "lulu",
					"resource": [1, 2, 3, 4],
					"skill": {
						"q": 1,
						"w": 3
					}
				}
			}`
		ifEqual, err := compareFullStateEqual(fullStateExpectedStr, fullState)
		assert.NoError(t, err)
		assert.True(t, ifEqual)
	})
}
