package jdelta

import (
	"errors"
)

type Executor interface {
	Set(value interface{}) error
	Clear() error
	Append(value interface{}) error
}

type mapExecutor struct {
	out map[string]interface{}
	key string
}

type arrayExecutor struct {
	out   []interface{}
	index int
}

func (m *mapExecutor) Set(value interface{}) error {
	(m.out)[m.key] = value
	return nil
}

func (m *mapExecutor) Clear() error {
	// delete value in a map
	if _, ok := (m.out)[m.key]; ok {
		delete(m.out, m.key)
	} else {
		return ErrKeyDoesNotExistInMap
	}
	return nil
}

func (m *mapExecutor) Append(value interface{}) error {
	if current, ok := ((m.out)[m.key]).([]interface{}); ok {
		if slicesToAppend, ok := value.([]interface{}); ok {
			updatedArray := append(current, slicesToAppend...)
			(m.out)[m.key] = updatedArray
		} else {
			return ErrValueToAppendMustBeArray
		}
	} else {
		if (m.out)[m.key] != nil {
			return ErrAppendToNonElement
		} else if slicesToAppend, ok := value.([]interface{}); ok {
			(m.out)[m.key] = slicesToAppend
		} else {
			return ErrValueToAppendMustBeArray
		}
	}
	return nil
}

func (a *arrayExecutor) Clear() error {
	// https://github.com/golang/go/wiki/SliceTricks
	// Remove a specific index in an arry has limitation due to my reads from golang docs above
	// so it is not recommended to remove a specific index in an array due to memory leak and hard to track state with null pointers
	// delete value in an array
	return ErrRemoveIndexInArrayNotSupported
}

func (a *arrayExecutor) Append(value interface{}) error {
	if len(a.out) <= a.index {
		return ErrIndexOutOfBound
	}

	if current, ok := ((a.out)[a.index]).([]interface{}); ok {
		if slicesToAppend, ok := value.([]interface{}); ok {
			updatedArray := append(current, slicesToAppend...)
			(a.out)[a.index] = updatedArray
		} else {
			return errors.New("value to append must be an array")
		}
	} else {
		return errors.New("unable to operate append action on array")
	}
	return nil
}

func (a *arrayExecutor) Set(value interface{}) error {
	// set value in an array
	if a.index >= len(a.out) {
		return ErrIndexOutOfBound
	}

	(a.out)[a.index] = value
	return nil
}
