package jdelta

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestExecutor(t *testing.T) {
	t.Run("should be able to replace a value to the same type", func(t *testing.T) {
		key := "damage_dealt"
		out := map[string]interface{}{key: 4396}
		mapExecutor := mapExecutor{
			out: out,
			key: key,
		}
		err := mapExecutor.Set(1000)
		assert.NoError(t, err)
		assert.Equal(t, out[key], 1000)
	})

	t.Run("should be able to remove a single value", func(t *testing.T) {
		key := "damage_dealt"
		out := map[string]interface{}{key: 4396}
		mapExecutor := mapExecutor{
			out: out,
			key: key,
		}
		err := mapExecutor.Clear()
		assert.Equal(t, out[key], nil)
		assert.NoError(t, err)
	})

	t.Run("should be able to replace a value to the different type", func(t *testing.T) {
		key := "damage_dealt"
		out := map[string]interface{}{key: "4396"}
		mapExecutor := mapExecutor{
			out: out,
			key: key,
		}
		err := mapExecutor.Set(1000)
		assert.NoError(t, err)
		assert.Equal(t, out[key], 1000)
	})

	t.Run("should be able to add a value", func(t *testing.T) {
		key := "damage_dealt"
		out := map[string]interface{}{}
		mapExecutor := mapExecutor{
			out: out,
			key: key,
		}
		err := mapExecutor.Set(1000)
		assert.NoError(t, err)
		assert.Equal(t, out[key], 1000)
	})

	t.Run("should throw error when trying to call append", func(t *testing.T) {
		key := "damage_dealt"
		out := map[string]interface{}{}
		mapExecutor := mapExecutor{
			out: out,
			key: key,
		}
		err := mapExecutor.Append(1000)
		assert.Error(t, err)
	})
}

func TestArrayExecutor(t *testing.T) {
	t.Run("should be able to set a value to a valid index", func(t *testing.T) {
		out := []interface{}{1, 2, 3}
		arrayExecutor := arrayExecutor{
			out:   out,
			index: 1,
		}
		err := arrayExecutor.Set(1000)
		assert.NoError(t, err)
		assert.Equal(t, out[1], 1000)
	})

	t.Run("should send an error to set a value to a not valid index", func(t *testing.T) {
		out := []interface{}{1, 2, 3}
		arrayExecutor := arrayExecutor{
			out:   out,
			index: 3,
		}
		err := arrayExecutor.Set(1000)
		assert.Error(t, err)
		assert.Equal(t, out[1], 2)
	})

	t.Run("should be able to append a value on array executor", func(t *testing.T) {
		out := []interface{}{1, 2, 3}
		m := make([]interface{}, 1)
		m[0] = out
		arrayExecutor := arrayExecutor{
			out:   m,
			index: 0,
		}
		err := arrayExecutor.Append([]interface{}{"4"})

		expectedArray := []interface{}{1, 2, 3, "4"}
		assert.NoError(t, err)
		assert.Equal(t, (arrayExecutor.out)[0], expectedArray)
	})

	t.Run("should be able to append a value on map executor", func(t *testing.T) {
		m := []interface{}{1, 2, 3}
		out := map[string]interface{}{
			"x": m,
		}
		mapExecutor := mapExecutor{
			out: out,
			key: "x",
		}
		err := mapExecutor.Append([]interface{}{"4"})

		expectedArray := []interface{}{1, 2, 3, "4"}
		assert.NoError(t, err)
		assert.Equal(t, (mapExecutor.out)["x"], expectedArray)
	})

	t.Run("should be able to append an array on array executor", func(t *testing.T) {
		out := []interface{}{1, 2, 3}
		m := make([]interface{}, 1)
		m[0] = out
		arrayExecutor := arrayExecutor{
			out:   m,
			index: 0,
		}
		err := arrayExecutor.Append([]interface{}{4, 5, 6})
		expectedArray := []interface{}{1, 2, 3, 4, 5, 6}
		assert.NoError(t, err)
		assert.Equal(t, (arrayExecutor.out)[0], expectedArray)
	})

	t.Run("should be able to append an array on map executor", func(t *testing.T) {
		m := []interface{}{1, 2, 3}
		out := map[string]interface{}{
			"x": m,
		}
		mapExecutor := mapExecutor{
			out: out,
			key: "x",
		}
		err := mapExecutor.Append([]interface{}{4, 5, 6})

		expectedArray := []interface{}{1, 2, 3, 4, 5, 6}
		assert.NoError(t, err)
		assert.Equal(t, (mapExecutor.out)["x"], expectedArray)
	})
}
