package jdelta

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestPatchDeltas(t *testing.T) {
	t.Run("Test empty delta operations", func(t *testing.T) {
		deltaStr := `[]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test adding entry with layers", func(t *testing.T) {
		deltaStr := `[["test", {"a": {"b": 1}}]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3
				}
			},
			"test": {"a": {"b": 1}}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test removing entries", func(t *testing.T) {
		deltaStr := `[["damage_dealt"]]`
		expectedFullStateStr := `{
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test removing entries in map", func(t *testing.T) {
		deltaStr := `[["champion.skill.q"]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test removing array", func(t *testing.T) {
		deltaStr := `[["champion.resource"]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test full replacement", func(t *testing.T) {
		deltaStr := `[[{"test": {"a": {"b": 1}}}]]`
		expectedFullStateStr := `{
			"test": {"a": {"b": 1}}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test replace an array field with new value", func(t *testing.T) {
		deltaStr := `[["position", {"early_game": [1,2]}]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": {"early_game": [1,2]},
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append array to existing array", func(t *testing.T) {
		deltaStr := `[["champion.resource", "a", [5,6]]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4, 5, 6],
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append complex items to existing array", func(t *testing.T) {
		deltaStr := `[["champion.resource", "a", [[1,2], 5, 6, "test"]]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4, [1,2], 5, 6, "test"],
				"skill": {
					"q": 1,
					"w": 3
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test adding entry with layers", func(t *testing.T) {
		deltaStr := `[["champion.skill.combo", [[1,2],[3,4]]]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3,
					"combo": [[1,2],[3,4]]
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test adding empty delta message", func(t *testing.T) {
		deltaStr := `[[]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3
				}
			},
			"test": {"a": {"b": 1}}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.Error(t, err)
		assert.Nil(t, f)
	})

	t.Run("Test adding entry with consecutive periods", func(t *testing.T) {
		deltaStr := `[["champion.skill..combo", "2"]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3,
					"combo": "2"
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.Error(t, err)
		assert.Nil(t, f)
	})

	t.Run("Test adding entry with empty brackets", func(t *testing.T) {
		deltaStr := `[["champion.skill[].combo", "2"]]`
		expectedFullStateStr := `{
			"damage_dealt": 12092,
			"position": [
				[1, 2],
				[3, 4]
			],
			"champion": {
				"name": "lulu",
				"resource": [1, 2, 3, 4],
				"skill": {
					"q": 1,
					"w": 3,
					"combo": "2"
				}
			}
		}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.Error(t, err)
		assert.Nil(t, f)
	})
}
