package jdelta

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestPatchDeltasArrayExample(t *testing.T) {
	t.Run("Test add example 1", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two[1]", 5]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 5]}]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test add example 2", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two[2]", 5]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrIndexOutOfBound)
	})

	t.Run("Test add example 3", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].three.baz", 5]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrInvalidMapExpectation)
	})

	t.Run("Test add example 4", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].three", 5]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4],"three": 5}]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test add example 5", func(t *testing.T) {
		deltaStr := `[["foo.bar", 5]]`
		expectedFullStateStr := `{"foo": {"bar": 5}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test add example 6", func(t *testing.T) {
		deltaStr := `[["foo.baz", 5]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4]}]],"baz": 5}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append example 1", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two[1]", "a", [5]]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.NotNil(t, err, ErrAppendToNonElement)
	})

	t.Run("Test append example 3", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two", "a", [5]]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4, 5]}]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append example 4", func(t *testing.T) {
		deltaStr := `[["foo.bar[0]", "a", [5, 6]]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4]},5,6]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append example 5", func(t *testing.T) {
		deltaStr := `[["foo.bar", "a", [5]]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4]}], 5]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append example 6", func(t *testing.T) {
		deltaStr := `[["foo.bar[0]", "a", 5]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrValueToAppendMustBeArray)
	})

	t.Run("Test append example 7", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][2]", "a", [5]]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrIndexOutOfBound)
	})

	t.Run("Test append example 8", func(t *testing.T) {
		deltaStr := `[["foo.bar", "a", []]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{"two": [3, 4]}]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test remove example 1", func(t *testing.T) {
		deltaStr := `[["foo"]]`
		expectedFullStateStr := `{}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test remove example 2", func(t *testing.T) {
		deltaStr := `[["foo.bar"]]`
		expectedFullStateStr := `{"foo": {}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test remove example 3", func(t *testing.T) {
		deltaStr := `[["foo.bar[0]"]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrRemoveIndexInArrayNotSupported)
	})

	t.Run("Test remove example 4", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1]"]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrRemoveIndexInArrayNotSupported)
	})

	t.Run("Test remove example 5", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][2]"]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrRemoveIndexInArrayNotSupported)
	})

	t.Run("Test remove example 6", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two[1]"]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrRemoveIndexInArrayNotSupported)
	})

	t.Run("Test remove example 7", func(t *testing.T) {
		deltaStr := `[["foo.baz"]]`

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, err, ErrKeyDoesNotExistInMap)
	})

	t.Run("Test remove example 8", func(t *testing.T) {
		deltaStr := `[["foo.bar[0][1].two"]]`
		expectedFullStateStr := `{"foo": {"bar": [[{"one": [1, 2]},{}]]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr2)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test replace deeper map", func(t *testing.T) {
		deltaStr := `[["test.a.b", 1]]`
		expectedFullStateStr := `{"test": {"a": {"b": 1, "x": 1}}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr3)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test replace deeper map 2", func(t *testing.T) {
		deltaStr := `[["test", {"a": {"b": 1}}]]`
		expectedFullStateStr := `{"test": {"a": {"b": 1}}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr3)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append fallback as add for array 1", func(t *testing.T) {
		deltaStr := `[["test.b", "a", [1,2]]]`
		expectedFullStateStr := `{"test": {"a": 3, "b": [1,2]}}`

		fullStateExpected, err := FullStateStringToMap(expectedFullStateStr)
		assert.NoError(t, err)
		assert.NotNil(t, fullStateExpected)

		fullState, err := FullStateStringToMap(fullStateStr4)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		f, err := PatchDeltas(fullState, deltas)
		assert.NoError(t, err)
		assert.Equal(t, fullStateExpected, f)
	})

	t.Run("Test append fallback as add for array 2", func(t *testing.T) {
		deltaStr := `[["test.a", "a", [1,2]]]`

		fullState, err := FullStateStringToMap(fullStateStr4)
		assert.NoError(t, err)

		deltas, err := DeltaStateStringToInterface(deltaStr)
		assert.NoError(t, err)

		_, err = PatchDeltas(fullState, deltas)
		assert.Equal(t, ErrAppendToNonElement, err)
	})
}
