package lifecycle

import (
	"fmt"
	"strings"
)

type ErrorList interface {
	error
	Append(...error)
	IsEmpty() bool
	Expand() []error
}

type errorList struct {
	errors []error
}

func CombineErrors(src, err error) error {
	if err == nil {
		return src
	}
	if src == nil {
		return err
	}
	if cast, ok := src.(ErrorList); ok {
		if cast2, ok := err.(ErrorList); ok {
			cast.Append(cast2.Expand()...)
		} else {
			cast.Append(err)
		}
		return cast
	}
	if cast, ok := err.(ErrorList); ok {
		cast.Append(src)
		return cast
	}
	return NewErrorList(src, err)
}

func (e *errorList) Append(errs ...error) {
	e.errors = append(e.errors, errs...)
}

func NewErrorList(errs ...error) ErrorList {
	return &errorList{errs}
}

func (e *errorList) Expand() []error {
	out := make([]error, len(e.errors))
	copy(out, e.errors)
	return out
}

func (e *errorList) IsEmpty() bool {
	return e.errors == nil
}

func (e *errorList) Error() string {
	length := len(e.errors)
	errStr := make([]string, length)
	for i, e := range e.errors {
		errStr[i] = e.Error()
	}
	suffix := ""
	if length > 0 {
		suffix = "\n"
	}
	return fmt.Sprintf("%d errors%s%s", length, suffix, strings.Join(errStr, "\n"))
}
