package lifecycle

import (
	"errors"
	"sync"
	"sync/atomic"
	"syscall"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestManager(t *testing.T) {
	expected := errors.New("expected error")

	t.Run("Should execute cleanly without hooks", func(t *testing.T) {
		assert.Empty(t, NewManager().ExecuteAll())
	})

	t.Run("Should ExecuteAll cleanly with clean hooks", func(t *testing.T) {
		m := NewManager()
		m.RegisterHook("test", func() error { return nil })
		m.RegisterHook("test2", func() error { return nil })
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should count ExecuteAll errors", func(t *testing.T) {
		m := NewManager()
		m.RegisterHook("test", func() error { return expected })
		m.RegisterHook("test2", func() error { return nil })
		m.RegisterHook("test3", func() error { return expected })
		assert.Equal(t, NewErrorList(expected, expected), m.ExecuteAll())
	})

	t.Run("Should reset count when called", func(t *testing.T) {
		m := NewManager()
		m.RegisterHook("test", func() error { return expected })
		assert.Equal(t, NewErrorList(expected), m.ExecuteAll())
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should report ExecuteAll errors to the Reporter", func(t *testing.T) {
		m := NewManager()
		var errored interface{}
		m.SetOnError(func(key interface{}, err error) { errored = key })
		m.RegisterHook("test", func() error { return expected })
		m.RegisterHook("test2", func() error { return nil })
		assert.Equal(t, NewErrorList(expected), m.ExecuteAll())
		assert.Equal(t, "test", errored)
	})

	t.Run("Should allow early execution of a hook", func(t *testing.T) {
		m := NewManager()
		var errored interface{}
		m.SetOnError(func(key interface{}, err error) { errored = key })
		m.RegisterHook("test", func() error { return expected })
		assert.Equal(t, expected, m.ExecuteHook("test"))
		assert.Equal(t, "test", errored)
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should ignore reexecution of a hook", func(t *testing.T) {
		m := NewManager()
		m.RegisterHook("test", func() error { return errors.New("broken") })
		assert.Equal(t, errors.New("broken"), m.ExecuteHook("test"))
		assert.Equal(t, nil, m.ExecuteHook("test"))
		m.RegisterHook("test", func() error { return errors.New("broken") })
		assert.Equal(t, errors.New("broken"), m.ExecuteHook("test"))
		assert.Equal(t, nil, m.ExecuteHook("test"))
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should allow reentry during hook execution", func(t *testing.T) {
		m := NewManager()
		m.RegisterHook("test", func() error { assert.NoError(t, m.ExecuteAll()); return nil })
		m.RegisterHook("test2", func() error { assert.NoError(t, m.ExecuteAll()); return expected })
		assert.Equal(t, NewErrorList(expected), m.ExecuteAll())
	})

	t.Run("Should execute hooks in FILO/stack order", func(t *testing.T) {
		m := NewManager()
		count := 0
		m.RegisterHook("test", func() error {
			assert.Equal(t, count, 1)
			count = count + 1
			return nil
		})
		m.RegisterHook("test2", func() error { return expected })
		assert.Equal(t, expected, m.ExecuteHook("test2"))
		m.RegisterHook("test3", func() error {
			assert.Equal(t, count, 0)
			count = count + 1
			return nil
		})
		m.RegisterHook("test4", func() error {
			return expected
		})
		assert.Equal(t, NewErrorList(expected), m.ExecuteAll())
	})

	t.Run("Should allow registration of background threads", func(t *testing.T) {
		m := NewManager()
		m.RunUntilComplete(func() {})
		assert.Nil(t, m.ExecuteAll())
	})

	t.Run("Should report panic in background threads", func(t *testing.T) {
		m := NewManager()
		var wg sync.WaitGroup
		m.SetOnPanic(func(src interface{}, err error) { wg.Done() })
		wg.Add(1)
		m.RunUntilComplete(func() { panic("boom") })
		wg.Wait()
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should block on background threads", func(t *testing.T) {
		m := NewManager()
		background := new(sync.WaitGroup)
		background.Add(1)
		m.RunUntilComplete(func() { background.Wait() })

		var completed uint32
		go func() {
			assert.NoError(t, m.ExecuteAll())
			atomic.StoreUint32(&completed, 1)
		}()
		time.Sleep(10 * time.Millisecond)
		assert.Empty(t, atomic.LoadUint32(&completed))
		background.Done()
		time.Sleep(10 * time.Millisecond)
		assert.NotEmpty(t, atomic.LoadUint32(&completed))
	})

	t.Run("Should allow registration of ticking jobs", func(t *testing.T) {
		m := NewManager()
		m.TickUntilClosed(func() {}, time.Nanosecond)
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should report panic in ticking jobs", func(t *testing.T) {
		m := NewManager()
		var wg sync.WaitGroup
		m.SetOnPanic(func(src interface{}, err error) { wg.Done() })
		wg.Add(1)
		m.TickUntilClosed(func() { panic("boom") }, time.Nanosecond)
		wg.Wait()
		assert.NoError(t, m.ExecuteAll())
	})

	t.Run("Should block on incomplete ticking jobs", func(t *testing.T) {
		m := NewManager()
		background := new(sync.WaitGroup)
		background.Add(1)

		m.TickUntilClosed(func() { background.Wait() }, time.Nanosecond)

		time.Sleep(time.Millisecond)

		var completed uint32
		go func() {
			assert.NoError(t, m.ExecuteAll())
			atomic.StoreUint32(&completed, 1)
		}()

		assert.Empty(t, atomic.LoadUint32(&completed))
		time.Sleep(10 * time.Millisecond)
		background.Done()
		time.Sleep(10 * time.Millisecond)
		assert.NotEmpty(t, atomic.LoadUint32(&completed))
	})

	t.Run("Should allow waiting for system calls", func(t *testing.T) {
		m := NewManager()
		var completed uint32
		go func() {
			assert.Equal(t, syscall.SIGUSR2, m.ListenForInterrupt())
			atomic.StoreUint32(&completed, 1)
		}()
		time.Sleep(10 * time.Millisecond)
		assert.Empty(t, atomic.LoadUint32(&completed))

		require.NoError(t, Interrupt())
		time.Sleep(10 * time.Millisecond)
		assert.NotEmpty(t, atomic.LoadUint32(&completed))
	})

	t.Run("Should allow waiting for completion (all threads finish)", func(t *testing.T) {
		m := NewManager()
		m.RunUntilComplete(func() {})
		m.TickUntilClosed(func() {}, time.Nanosecond).Close()
		m.WaitForCompletion(time.Now().Add(time.Hour))
	})

	t.Run("Should allow waiting for completion (time expired)", func(t *testing.T) {
		m := NewManager()
		run := int32(1)
		m.RunUntilComplete(func() {
			for atomic.LoadInt32(&run) != 0 {
			}
			assert.NoError(t, m.ExecuteAll())
		})
		m.TickUntilClosed(func() {}, time.Second)
		m.WaitForCompletion(time.Now().Add(time.Millisecond))
		atomic.StoreInt32(&run, 0)
	})
}
