// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

import { EcrRepositoryStack } from './stacks/ecr-repository';
import { ECSClusterStack } from './stacks/ecs-cluster';
import { ECSFargateServiceStack } from './stacks/ecs-fargate-service';
import { VpcStack } from './stacks/vpc';

interface EnvProps {
  prefix: string;
  env: cdk.Environment;
  ec2AsgInstanceType: ec2.InstanceType;
  ec2AsgCapacity: number;
  vpcId: string;
  vpcPrivSubnets: ec2.SubnetAttributes[];
}

const app = new cdk.App();

createStacks(app, {
  prefix: 'Dev',
  env: {
    account: '797743463538', // e2-ingest
    region: 'us-west-2',
  },
  ec2AsgInstanceType: new ec2.InstanceType('m5.large'),
  ec2AsgCapacity: 0, // used for dev02 test instance
  vpcId: 'vpc-00bb9e143bf9366c8', // e2-ingest vpc
  vpcPrivSubnets: [
    { subnetId: 'subnet-04268460eab6f80ce', availabilityZone: 'us-west-2a', routeTableId: 'rtb-09f97cada3fca72a9' }, // 10.206.144.0/24 // e2-ingest private vpc us-west-2a
    { subnetId: 'subnet-00aa0035c7b33c837', availabilityZone: 'us-west-2b', routeTableId: 'rtb-0c44634fcda39ed7c' }, // 10.206.145.0/24 // e2-ingest private vpc us-west-2b
    { subnetId: 'subnet-013355186d0721c8a', availabilityZone: 'us-west-2c', routeTableId: 'rtb-0141745c2916e0824' }, // 10.206.146.0/24 // e2-ingest private vpc us-west-2c
  ]
});

// createStacks for each environment
function createStacks(scope: cdk.App, props: EnvProps) {
  const { prefix, env } = props;

  const vpcStack = new VpcStack(scope, prefix + 'Vpc', {
    env,
    vpcId: props.vpcId,
    vpcPrivSubnets: props.vpcPrivSubnets,
  });

  const ecrRepoStack = new EcrRepositoryStack(app, prefix + 'LoadtesterRepo', { env, repositoryName: 'loadtester' }); // repo name as used by scripts/docker_push.sh

  const clusterStack = new ECSClusterStack(app, prefix + 'LoadtesterECSCluster', {
    env,
    vpc: vpcStack.vpc,
    ec2Asg: {
      instanceType: props.ec2AsgInstanceType,
      minCapacity: props.ec2AsgCapacity,
      maxCapacity: props.ec2AsgCapacity,
    },
  });

  new ECSFargateServiceStack(app, prefix + 'LoadtesterService', {
    env,
    vpc: vpcStack.vpc,
    cluster: clusterStack.cluster,
    ecrRepo: ecrRepoStack.ecrRepo
  });
}