import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';

interface ECSServiceStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
  cluster: ecs.ICluster;
  ecrRepo: ecr.IRepository;
}

export class ECSEC2ServiceStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ECSServiceStackProps) {
    super(scope, id, props);

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDefinition', {
      networkMode: ecs.NetworkMode.AWS_VPC,
    });

    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    const container = taskDefinition.addContainer('TaskContainer', LoadtestContainerDefinition(props));
    container.addUlimits({
      name: ecs.UlimitName.NOFILE, // number of open files, needed to handle socket connections
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Service
    new ecs.Ec2Service(this, 'TaskService', {
      serviceName: `load_tester`,
      cluster: props.cluster, // cluster must have an ASG with enough capacity
      taskDefinition,
      assignPublicIp: false,
      desiredCount: 1, // updated through /scripts/run.sh
      placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
    });
  }
}

export function LoadtestContainerDefinition(props: ECSServiceStackProps): ecs.ContainerDefinitionOptions {
  return {
      image: ecs.ContainerImage.fromEcrRepository(props.ecrRepo, 'latest'), // Requires props.ecrRepo to exist and have a registered docker image
      logging: new ecs.AwsLogDriver({ streamPrefix: 'e2topics' }),
      memoryReservationMiB: 1024, // 1 GB
      environment: {
        TARGETENV: "dev",
      }
  };
}
