import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';
import * as route53 from '@aws-cdk/aws-route53';

interface VpcStackProps extends cdk.StackProps {
  vpcId: string; // id to import the vpc
  vpcPrivSubnets: ec2.SubnetAttributes[]; // to import the 3 private subnets (one for each availablility zone)
}

export class VpcStack extends cdk.Stack {
  public readonly vpc: ec2.IVpc;
  public readonly vpcPrivSubnets: ec2.ISubnet[];

  constructor(scope: cdk.Construct, name: string, props: VpcStackProps) {
    super(scope, name, props);

    // Import already existing VPC
    this.vpc = ec2.Vpc.fromLookup(this, 'Vpc', { vpcId: props.vpcId });

    // In theory, Vpc.fromLookup should also import the subnets,
    // but for some reason this.vpc.privateSubnets is empty.
    // Import the subnets manually and expose them on a stack property.
    this.vpcPrivSubnets = props.vpcPrivSubnets.map((subnetAttrs, i) =>
      ec2.Subnet.fromSubnetAttributes(this, `PrivSubnet${i}`, subnetAttrs)
    );

    const domainName = 'staging.connections.twitch.a2z.com'; // determined by origin of the service endpoint
    const endpoint = new ec2.InterfaceVpcEndpoint(this, 'Endpoint', {
      vpc: this.vpc,
      service: {
        name: 'com.amazonaws.vpce.us-west-2.vpce-svc-07d8c2533f6d96894',
        port: 443,
      },
      privateDnsEnabled: false
    });
    const zone = new route53.PrivateHostedZone(this, 'Zone', {
      zoneName: domainName,
      vpc: this.vpc,
    });

    new route53.ARecord(this, 'CName', {
      zone,
      target: route53.RecordTarget.fromAlias(new InterfaceVpcEndpointTarget(endpoint)),
    });

    const serviceSecGrp = new ec2.SecurityGroup(
      this,
      "loadtestServiceSecurityGroup",
      {
        allowAllOutbound: true,
        securityGroupName: "loadtestServiceSecGrp",
        vpc: this.vpc,
      }
    );

    serviceSecGrp.connections.allowTo(
      endpoint, ec2.Port.tcp(443), 
    )
  }
}

class InterfaceVpcEndpointTarget implements route53.IAliasRecordTarget {
  constructor(readonly endpoint: ec2.InterfaceVpcEndpoint) {}

  // select the first DNS entry an extract host zone + dns name from it
  public bind(): route53.AliasRecordTargetConfig {
    const parts = cdk.Fn.split(':', cdk.Fn.select(0, this.endpoint.vpcEndpointDnsEntries));
    return {
      hostedZoneId: cdk.Fn.select(0, parts),
      dnsName: cdk.Fn.select(1, parts),
    };
  }
}