package connections

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/foundation/twitchclient"
)

// RiotUser specifies the fields of a Riot connection
type RiotUser struct {
	TwitchID     string     `json:"twitch_id"`
	RiotPUUID    string     `json:"riot_id"`
	AccessToken  string     `json:"access_token,omitempty"`
	RefreshToken string     `json:"refresh_token,omitempty"`
	ExpiresOn    *time.Time `json:"expires_on,omitempty"`
	CreatedOn    *time.Time `json:"created_on,omitempty"`
	UpdatedOn    *time.Time `json:"updated_on,omitempty"`
}

// RiotConnections encodes a list of riot connections
type RiotConnections struct {
	Connected []RiotUser `json:"connected"`
}

func (c *client) AuthRiot(ctx context.Context, uid string, values url.Values, opts *twitchclient.ReqOpts) (string, error) {
	u := &url.URL{Path: fmt.Sprintf("/v2/%s/riot/auth", uid), RawQuery: values.Encode()}
	res, err := c.do(ctx, "GET", u.String(), "authRiot", opts)
	if err != nil {
		return "", err
	}
	defer res.Body.Close()

	if res.StatusCode != http.StatusOK {
		return "", decodeError(res)
	}

	var resp RedirectPathResp
	if err := json.NewDecoder(res.Body).Decode(&resp); err != nil {
		return "", err
	}
	return resp.Path, nil
}

func (c *client) CallbackRiot(ctx context.Context, uid string, values url.Values, opts *twitchclient.ReqOpts) error {
	u := &url.URL{Path: fmt.Sprintf("/v2/%s/riot/callback", uid), RawQuery: values.Encode()}
	res, err := c.do(ctx, "GET", u.String(), "callbackRiot", opts)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	if res.StatusCode != http.StatusCreated {
		return decodeError(res)
	}
	return nil
}

func (c *client) GetRiotUser(ctx context.Context, uid string, opts *twitchclient.ReqOpts) (*RiotUser, error) {
	u := &url.URL{Path: fmt.Sprintf("/v2/%s/riot", uid)}
	res, err := c.do(ctx, "GET", u.String(), "getRiotUser", opts)
	if err != nil {
		return nil, err
	}
	defer res.Body.Close()

	switch res.StatusCode {
	case http.StatusOK:
		var data RiotUser
		err = json.NewDecoder(res.Body).Decode(&data)
		return &data, err
	case http.StatusNotFound:
		return nil, ErrNoConnection
	default:
		return nil, decodeError(res)
	}
}

func (c *client) DeleteRiotUser(ctx context.Context, uid string, opts *twitchclient.ReqOpts) error {
	u := &url.URL{Path: fmt.Sprintf("/v2/%s/riot", uid)}
	res, err := c.do(ctx, "DELETE", u.String(), "deleteRiotUser", opts)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	if res.StatusCode != http.StatusNoContent {
		return decodeError(res)
	}
	return nil
}

func (c *client) GetRiotConnected(ctx context.Context, riotID string, opts *twitchclient.ReqOpts) (*RiotUser, error) {
	u := &url.URL{Path: "/v2/riot"}

	postData := ConnectedReq{IDs: []string{riotID}}
	body, err := json.Marshal(postData)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", u.String(), bytes.NewReader(body))
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")

	merged := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       fmt.Sprintf("service.connections.%s", "getRiotConnected"),
		StatSampleRate: defaultStatSampleRate,
	})

	res, err := c.Do(ctx, req, merged)
	if err != nil {
		return nil, err
	}
	defer res.Body.Close()

	switch res.StatusCode {
	case http.StatusOK:
		var data RiotConnections
		err = json.NewDecoder(res.Body).Decode(&data)
		if err != nil {
			return nil, err
		}
		if len(data.Connected) < 1 {
			return nil, ErrNoConnection
		}
		fbConnection := data.Connected[0]
		return &fbConnection, nil
	default:
		return nil, decodeError(res)
	}
}

func (c *client) AdminCreateRiot(ctx context.Context, params RiotUser, opts *twitchclient.ReqOpts) error {
	u := &url.URL{Path: "/admin/v2/riot"}

	body, err := json.Marshal(params)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("POST", u.String(), bytes.NewReader(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	merged := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       fmt.Sprintf("service.connections.%s", "adminCreateRiot"),
		StatSampleRate: defaultStatSampleRate,
	})

	res, err := c.Do(ctx, req, merged)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	switch res.StatusCode {
	case http.StatusCreated:
		return nil
	default:
		return decodeError(res)
	}
}

func (c *client) AdminDeleteRiot(ctx context.Context, uid, riotID string, opts *twitchclient.ReqOpts) error {
	u := &url.URL{Path: "/admin/v2/riot"}
	query := u.Query()
	query.Set("user_id", uid)
	query.Set("riot_id", riotID)
	u.RawQuery = query.Encode()

	req, err := c.NewRequest("DELETE", u.String(), nil)
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	merged := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       fmt.Sprintf("service.connections.%s", "adminDeleteRiot"),
		StatSampleRate: defaultStatSampleRate,
	})

	res, err := c.Do(ctx, req, merged)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	switch res.StatusCode {
	case http.StatusNoContent:
		return nil
	default:
		return decodeError(res)
	}
}
