package dataservice

import (
	"fmt"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/go-redis/redis"
)

type DataServiceImpl struct {
	KinesisClient  KinesisAPI
	RedisClient    RedisAPI
	DynamoDBClient DynamoDBAPI
}

type redisAPIImpl struct {
	redisClient *redis.Client
}

func (r *redisAPIImpl) LRange(key string, start int64, end int64) ([]string, error) {
	return r.redisClient.LRange(key, start, end).Result()
}

func (r *redisAPIImpl) RPush(key string, values ...string) error {
	return r.redisClient.RPush(key, values).Err()
}

func (r *redisAPIImpl) Expire(key string, expiration time.Duration) error {
	return r.redisClient.Expire(key, expiration).Err()
}

type kinesisAPIImpl struct {
	kinesisClient KinesisAPI
}

func (k *kinesisAPIImpl) ListShards(input *kinesis.ListShardsInput) (*kinesis.ListShardsOutput, error) {
	return k.kinesisClient.ListShards(input)
}

type dynamoDBAPIImpl struct {
	dynamoDBClient DynamoDBAPI
}

func (d *dynamoDBAPIImpl) GetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	return d.dynamoDBClient.GetItem(input)
}

const (
	tableName                = "GameStream"
	shardListCachePrefix     = "shard-list-for-game-"
	shardListExpirationInSec = 300 // 5 min
)

func initRedisClient() *redisAPIImpl {
	return &redisAPIImpl{
		redisClient: redis.NewClient(&redis.Options{
			Addr: fmt.Sprintf("%s:6379", os.Getenv("redis_endpoint")),
		}),
	}
}

func initDynamoDBClient(s *session.Session) *dynamodb.DynamoDB {
	return dynamodb.New(s, &aws.Config{
		Region: aws.String(awsRegion),
	})
}

func initKinesisClient(s *session.Session) *kinesis.Kinesis {
	return kinesis.New(s, &aws.Config{Region: aws.String(awsRegion)})
}

func InitDataService() DataServiceAPI {
	s := session.Must(session.NewSessionWithOptions(session.Options{
		SharedConfigState: session.SharedConfigEnable,
	}))
	return &DataServiceImpl{
		KinesisClient:  initKinesisClient(s),
		RedisClient:    initRedisClient(),
		DynamoDBClient: initDynamoDBClient(s),
	}
}

func (ds *DataServiceImpl) GetStreamsForGame(gameID string) (*StreamList, error) {
	result, err := ds.DynamoDBClient.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(tableName),
		Key: map[string]*dynamodb.AttributeValue{
			"game_id": {
				S: aws.String(gameID),
			},
		},
	})
	if err != nil {
		return nil, err
	}

	if result == nil {
		return nil, nil
	}

	gs := GameStream{}
	err = dynamodbattribute.UnmarshalMap(result.Item, &gs)
	var streams []*string
	if len(gs.MainStreamARN) > 0 {
		mainStreamName := getStreamNameFromStreamARN(gs.MainStreamARN)
		streams = append(streams, &mainStreamName)
	}
	for _, streamARN := range gs.CloneStreamARNs {
		streamName := getStreamNameFromStreamARN(streamARN)
		streams = append(streams, &streamName)
	}
	return &StreamList{
		Value: streams,
	}, nil
}

func (ds *DataServiceImpl) ListShards(streamName *string, nextToken *string, shardArray []*kinesis.Shard) ([]*kinesis.Shard, error) {
	input := &kinesis.ListShardsInput{
		StreamName: streamName,
	}
	if nextToken != nil {
		input.NextToken = nextToken
	}

	output, err := ds.KinesisClient.ListShards(input)
	if err != nil {
		return nil, err
	}
	if output == nil || output.Shards == nil {
		return []*kinesis.Shard{}, nil
	}

	array := append(shardArray, output.Shards...)

	if output.NextToken != nil && len(output.Shards) > 0 {
		return ds.ListShards(streamName, output.NextToken, array)
	}
	return array, nil
}

func (ds *DataServiceImpl) GetShardID(gameID string, broadcasterID string) (*string, error) {
	streamName := getOriginalStreamNameByGameID(gameID)
	redisKey := fmt.Sprintf("%s%s", shardListCachePrefix, gameID)
	// hash keys in increasing order
	shardStartHashKeys, err := ds.RedisClient.LRange(redisKey, 0, -1)
	if err != nil && err != redis.Nil {
		return nil, err
	}
	if err == redis.Nil || shardStartHashKeys == nil || len(shardStartHashKeys) == 0 {
		// need to load shards into redis
		shardStartHashKeysPtr, err := ds.saveShardIDListIntoRedis(streamName, redisKey)
		if err != nil {
			return nil, err
		}
		shardStartHashKeys = *shardStartHashKeysPtr
	}
	shardID := findShardIDByBroadcasterID(&shardStartHashKeys, broadcasterID)
	return shardID, nil
}

func (ds *DataServiceImpl) saveShardIDListIntoRedis(streamName string, redisKey string) (*[]string, error) {
	shards, err := ds.ListShards(&streamName, nil, []*kinesis.Shard{})
	if err != nil {
		return nil, err
	}
	var shardStartHashKeys []string
	for _, shard := range shards {
		shardStartHashKeys = append(shardStartHashKeys, *shard.HashKeyRange.StartingHashKey)
	}

	err = ds.RedisClient.RPush(redisKey, shardStartHashKeys...)
	if err != nil {
		return nil, err
	}
	err = ds.RedisClient.Expire(redisKey, shardListExpirationInSec*time.Second)
	if err != nil {
		return nil, err
	}
	return &shardStartHashKeys, nil
}
