package dataservice_test

import (
	"fmt"
	"testing"

	"github.com/aws/aws-sdk-go/service/kinesis"

	data "code.justin.tv/devhub/mdaas-controller/stream-info/dataservice"
	fakes "code.justin.tv/devhub/mdaas-controller/stream-info/dataservice/dataservicefakes"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/stretchr/testify/assert"
)

const (
	testGameID        = "1234"
	testBroadcasterID = "41516171"
)

func initTestDataService() (*fakes.FakeKinesisAPI, *fakes.FakeDynamoDBAPI, *fakes.FakeRedisAPI, data.DataServiceAPI) {
	mockKinesis := &fakes.FakeKinesisAPI{}
	mockDynamoDB := &fakes.FakeDynamoDBAPI{}
	mockRedis := &fakes.FakeRedisAPI{}
	return mockKinesis, mockDynamoDB, mockRedis, &data.DataServiceImpl{
		KinesisClient:  mockKinesis,
		DynamoDBClient: mockDynamoDB,
		RedisClient:    mockRedis,
	}
}

func formatStreamNameByGameID(gameID string, index int) string {
	return fmt.Sprintf("game.%s.%d", gameID, index)
}

func formatStreamARNByStreamName(streamName string) string {
	return fmt.Sprintf("arn:aws:kinesis:us-west-2:444555666:stream/%s", streamName)
}

func mockShard() *kinesis.Shard {
	return &kinesis.Shard{
		ShardId: aws.String("shardId-000000000000"),
		HashKeyRange: &kinesis.HashKeyRange{
			StartingHashKey: aws.String("0"),
			EndingHashKey:   aws.String("340282366920938463463374607431768211455"),
		},
		SequenceNumberRange: &kinesis.SequenceNumberRange{
			StartingSequenceNumber: aws.String("49598450486769088894492668750083354095827565453254852610"),
		},
	}
}

func TestGetStreamForGame_no_record_in_dynamodb(t *testing.T) {
	_, mockDynamoDB, _, mockDataService := initTestDataService()
	mockDynamoDB.GetItemReturns(nil, nil)
	list, err := mockDataService.GetStreamsForGame(testGameID)
	assert.Nil(t, list)
	assert.Nil(t, err)
}

func TestGetStreamForGame_dynamodb_has_record(t *testing.T) {
	_, mockDynamoDB, _, mockDataService := initTestDataService()
	itemMap := map[string]*dynamodb.AttributeValue{
		"game_id": &dynamodb.AttributeValue{
			S: aws.String(testGameID),
		},
		"main_stream_arn": &dynamodb.AttributeValue{
			S: aws.String(formatStreamNameByGameID(testGameID, 1)),
		},
		"clone_stream_arns": &dynamodb.AttributeValue{
			SS: aws.StringSlice([]string{formatStreamNameByGameID(testGameID, 2), formatStreamNameByGameID(testGameID, 3)}),
		},
	}
	mockGetItemOutput := dynamodb.GetItemOutput{
		Item: itemMap,
	}
	mockDynamoDB.GetItemReturns(&mockGetItemOutput, nil)
	list, err := mockDataService.GetStreamsForGame(testGameID)
	assert.Equal(t, 3, len(list.Value))
	assert.Equal(t, "game.1234.1", *list.Value[0])
	assert.Equal(t, "game.1234.2", *list.Value[1])
	assert.Equal(t, "game.1234.3", *list.Value[2])
	assert.Nil(t, err)
}

func TestListShard_no_shard(t *testing.T) {
	mockKinesis, _, _, mockDataService := initTestDataService()
	mockKinesis.ListShardsReturns(nil, nil)
	streamName := formatStreamNameByGameID(testGameID, 1)
	shardArray, err := mockDataService.ListShards(&streamName, nil, []*kinesis.Shard{})
	assert.Empty(t, shardArray)
	assert.Nil(t, err)
}

func TestListShard_has_shard(t *testing.T) {
	mockKinesis, _, _, mockDataService := initTestDataService()
	ms := mockShard()
	mockOutput := &kinesis.ListShardsOutput{
		NextToken: nil,
		Shards:    []*kinesis.Shard{ms},
	}
	mockKinesis.ListShardsReturns(mockOutput, nil)
	streamName := formatStreamNameByGameID(testGameID, 1)
	shardArray, err := mockDataService.ListShards(&streamName, nil, []*kinesis.Shard{})
	assert.Equal(t, 1, len(shardArray))
	assert.Nil(t, err)
}

func TestGetShardID_no_record_in_redis(t *testing.T) {
	mockKinesis, _, mockRedis, mockDataService := initTestDataService()
	ms := mockShard()
	mockOutput := &kinesis.ListShardsOutput{
		NextToken: nil,
		Shards:    []*kinesis.Shard{ms},
	}
	mockKinesis.ListShardsReturns(mockOutput, nil)
	shardID, err := mockDataService.GetShardID(testGameID, testBroadcasterID)
	assert.Equal(t, 1, mockRedis.LRangeCallCount())
	assert.Equal(t, 1, mockKinesis.ListShardsCallCount())
	assert.Equal(t, 1, mockRedis.RPushCallCount())
	assert.Equal(t, 1, mockRedis.ExpireCallCount())
	assert.Nil(t, err)
	assert.Equal(t, "shardId-000000000000", *shardID)
}

func TestGetShardID_has_record_in_redis(t *testing.T) {
	mockKinesis, _, mockRedis, mockDataService := initTestDataService()
	ms := mockShard()
	mockOutput := &kinesis.ListShardsOutput{
		NextToken: nil,
		Shards:    []*kinesis.Shard{ms},
	}
	mockRedis.LRangeReturns([]string{"0"}, nil)
	mockKinesis.ListShardsReturns(mockOutput, nil)
	shardID, err := mockDataService.GetShardID(testGameID, testBroadcasterID)
	assert.Equal(t, 1, mockRedis.LRangeCallCount())
	assert.Equal(t, 0, mockKinesis.ListShardsCallCount())
	assert.Equal(t, 0, mockRedis.RPushCallCount())
	assert.Equal(t, 0, mockRedis.ExpireCallCount())
	assert.Nil(t, err)
	assert.Equal(t, "shardId-000000000000", *shardID)
}
