package dataservice

import (
	"time"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

//go:generate go run github.com/maxbrunsfeld/counterfeiter -generate

// StreamDetails is the struct of resposne for /streams/details
type StreamDetails struct {
	Shard *string `json:"shard"`
	// StreamARN  *string `json:"stream_arn"`
	StreamName *string `json:"stream_name"`
	// StreamStatus *string `json:"stream_status"`
}

// StreamList returns all streams for a specific gameID
type StreamList struct {
	Value []*string `json:"streams"`
}

// CloneStreamArnsType stands for string set
type CloneStreamArnsType []string

// GameStream is the entity definition when interacting with DynamoDB
type GameStream struct {
	CloneStreamARNs CloneStreamArnsType `json:"clone_stream_arns"`
	MainStreamARN   string              `json:"main_stream_arn"`
	GameID          string              `json:"game_id"`
}

//counterfeiter:generate . DataServiceAPI
type DataServiceAPI interface {
	GetStreamsForGame(gameID string) (*StreamList, error)
	ListShards(streamName *string, nextToken *string, shardArray []*kinesis.Shard) ([]*kinesis.Shard, error)
	GetShardID(gameID string, broadcasterID string) (*string, error)
}

// RedisAPI is the wrapper for redis API
//counterfeiter:generate . RedisAPI
type RedisAPI interface {
	LRange(key string, start int64, end int64) ([]string, error)
	RPush(key string, values ...string) error
	Expire(key string, expiration time.Duration) error
}

//counterfeiter:generate . DynamoDBAPI
type DynamoDBAPI interface {
	GetItem(*dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error)
}

//counterfeiter:generate . KinesisAPI
type KinesisAPI interface {
	ListShards(*kinesis.ListShardsInput) (*kinesis.ListShardsOutput, error)
}

const (
	awsRegion = "us-west-2"
)
