package dataservice

import (
	"crypto/md5"
	"encoding/hex"
	"fmt"
	"math/big"
	"strings"
)

const (
	maxHashKey = "340282366920938463463374607431768211455"
)

func compareHash(hash1 string, hash2 string) int {
	var longerHash = hash1
	var shorterHash = hash2
	if len(hash1) < len(hash2) {
		longerHash = hash2
		shorterHash = hash1
	}
	// the size of the longest hash key is 39
	extendedShorterHash := fmt.Sprintf("%039s", shorterHash)
	if longerHash > extendedShorterHash {
		if longerHash == hash1 {
			return 1
		}
		return -1
	} else if longerHash < extendedShorterHash {
		if longerHash == hash1 {
			return -1
		}
		return 1
	}
	return 0
}

// getHashForBroadcasterID will return a 128 bit integer as a string
func getHashForBroadcasterID(broadcasterID string) string {
	// This is an implementation of how Kinesis handles hashing, and should be replaced if we manually control buckets for broadcaster ids
	hashkey := big.NewInt(0)
	md5ForID := md5.New()
	md5ForID.Write([]byte(broadcasterID))
	hexString := hex.EncodeToString(md5ForID.Sum(nil))
	hashkey.SetString(hexString, 16)
	return hashkey.String()
}

// original stream will be named as "game.<game_id>.1"
func getOriginalStreamNameByGameID(gameID string) string {
	return fmt.Sprintf("game.%s.1", gameID)
}

func getStreamNameFromStreamARN(streamARN string) string {
	index := strings.Index(streamARN, "/")
	return streamARN[index+1:]
}

func formatShardIDfromIndex(index int) string {
	return fmt.Sprintf("shardId-%012d", index)
}

func findShardIDByBroadcasterID(shardStartHashKeys *[]string, broadcasterID string) *string {
	if shardStartHashKeys == nil {
		return nil
	}
	targetHash := getHashForBroadcasterID(broadcasterID)
	index := binarySearchIndex(shardStartHashKeys, &targetHash, 0, len(*shardStartHashKeys)-1)
	shardID := formatShardIDfromIndex(index)
	return &shardID
}

func binarySearchIndex(shardStartHashKeys *[]string, targetHash *string, startIndex int, endIndex int) int {
	mid := startIndex + (endIndex-startIndex)/2
	array := *shardStartHashKeys
	midVal := array[mid]
	cmp := compareHash(midVal, *targetHash)
	if cmp == 0 {
		return mid
	} else if cmp == 1 {
		return binarySearchIndex(shardStartHashKeys, targetHash, startIndex, mid-1)
	} else {
		var valAfterMid = maxHashKey
		if mid+1 < len(*shardStartHashKeys) {
			valAfterMid = array[mid+1]
		}
		if compareHash(valAfterMid, *targetHash) > 0 {
			return mid
		}
		return binarySearchIndex(shardStartHashKeys, targetHash, mid+1, endIndex)
	}
}
