package main

import (
	"encoding/json"
	"math/rand"
	"net/http"

	data "code.justin.tv/devhub/mdaas-controller/stream-info/dataservice"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
)

// Max consumers per stream var?
// Number of clones per game var?
// Override to give specific game additional clones?
const (
	delimiter             = "."
	numberOfClonedStreams = 5
)

const (
	ErrMsgHTTPMethodNotAllowed = "Method Not Allowed"
	ErrMsgInvalidGameID        = "game_id is invalid"
	ErrMsgInvalidBroadcasterID = "broadcaster_id is invalid"
	ErrMsgResouceNotFound      = "Resouce not found"
	ErrMsgShardRecordNotFound  = "Record of shard for the queried broadcaster_id is not found"
)

func getStreamDetailsForGame(game string, broadcasterID string, dataService data.DataServiceAPI) (*data.StreamDetails, error) {
	shardID, err := dataService.GetShardID(game, broadcasterID)
	if err != nil {
		return nil, err
	}
	if shardID == nil {
		return nil, nil
	}

	streams, err := dataService.GetStreamsForGame(game)
	if err != nil {
		return nil, err
	}
	if streams == nil {
		return nil, nil
	}

	randNumber := rand.Intn(len(streams.Value))
	stream := streams.Value[randNumber]

	return &data.StreamDetails{
		StreamName: stream,
		Shard:      shardID,
	}, nil
}

// handles request through /streams, return http code and response body
func HandleRequestForStreamsDetails(gameID string, broadcasterID string, dataService data.DataServiceAPI) (int, *string) {
	gameStream, err := getStreamDetailsForGame(gameID, broadcasterID, dataService)
	if err != nil || gameStream == nil {
		// For now, assume this is due to no content
		var errMsg = ErrMsgShardRecordNotFound
		if err != nil {
			errMsg = err.Error()
		}
		return http.StatusNotFound, &errMsg
	}

	gameStreamJSON, err := json.Marshal(*gameStream)
	if err != nil {
		return http.StatusInternalServerError, nil
	}
	gameStreamStr := string(gameStreamJSON)
	return http.StatusOK, &gameStreamStr
}

func HandleRequestForStreamNames(gameID string, dataService data.DataServiceAPI) (int, *string) {
	streams, err := dataService.GetStreamsForGame(gameID)
	if err != nil {
		return http.StatusNoContent, nil
	}

	if streams == nil {
		return http.StatusOK, nil
	}

	streamsJSON, err := json.Marshal(*streams)
	if err != nil {
		return http.StatusInternalServerError, nil
	}
	streamsStr := string(streamsJSON)

	return http.StatusOK, &streamsStr
}

// HandleRequest - request handler for 2 APIs:
// /streams/name?game=<game_id> - this one returns an array of stream names
// /streams/details?game=<game_id>&broadcaster_id=<broadcaster_id> - this one returns info the consumer will need to read from kinesis
func HandleRequest(request events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	if request.HTTPMethod != "GET" {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusMethodNotAllowed, Body: ErrMsgHTTPMethodNotAllowed}, nil
	}

	gameID, gameParamExists := request.QueryStringParameters["game"]
	if !gameParamExists || len(gameID) == 0 {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusBadRequest, Body: ErrMsgInvalidGameID}, nil
	}

	dataService := data.InitDataService()
	var respCode = http.StatusOK
	var respBody *string
	switch request.Resource {
	case "/streams/details":
		broadcasterID, broadcasterIDExists := request.QueryStringParameters["broadcaster_id"]
		if !broadcasterIDExists || len(broadcasterID) == 0 {
			return events.APIGatewayProxyResponse{StatusCode: http.StatusBadRequest, Body: ErrMsgInvalidBroadcasterID}, nil
		}
		respCode, respBody = HandleRequestForStreamsDetails(gameID, broadcasterID, dataService)
	case "/streams/name":
		respCode, respBody = HandleRequestForStreamNames(gameID, dataService)
	default:
		// do nothing
		return events.APIGatewayProxyResponse{StatusCode: http.StatusNotFound, Body: ErrMsgResouceNotFound}, nil
	}

	response := events.APIGatewayProxyResponse{StatusCode: respCode}
	if respBody != nil {
		response.Body = *respBody
	}
	return response, nil
}

func main() {
	lambda.Start(HandleRequest)
}
