package main_test

import (
	"encoding/json"
	"net/http"
	"strings"
	"testing"

	"github.com/aws/aws-sdk-go/aws"

	mainPkg "code.justin.tv/devhub/mdaas-controller/stream-info"
	data "code.justin.tv/devhub/mdaas-controller/stream-info/dataservice"
	fakes "code.justin.tv/devhub/mdaas-controller/stream-info/dataservice/dataservicefakes"

	"github.com/aws/aws-lambda-go/events"
	"github.com/stretchr/testify/assert"
)

func initTestDataService() *fakes.FakeDataServiceAPI {
	return &fakes.FakeDataServiceAPI{}
}

const (
	resourceForDetails = "/streams/details"
	resourceForNames   = "/streams/name"
	testGameID         = "1234"
	testBroadcasterID  = "41516161"
	testShardID        = "shardId-000000000000"
)

func TestRequestHandler_Invalid_http_method(t *testing.T) {
	badReq := events.APIGatewayProxyRequest{
		HTTPMethod: "POST",
	}
	resp, err := mainPkg.HandleRequest(badReq)
	assert.Equal(t, resp.Body, mainPkg.ErrMsgHTTPMethodNotAllowed)
	assert.Equal(t, resp.StatusCode, http.StatusMethodNotAllowed)
	assert.Nil(t, err)
}

func TestRequestHandler_Invalid_param_game(t *testing.T) {
	qp := map[string]string{
		"game": "",
	}
	badReq := events.APIGatewayProxyRequest{
		HTTPMethod:            "GET",
		QueryStringParameters: qp,
	}
	resp, err := mainPkg.HandleRequest(badReq)
	assert.Equal(t, resp.Body, mainPkg.ErrMsgInvalidGameID)
	assert.Equal(t, resp.StatusCode, http.StatusBadRequest)
	assert.Nil(t, err)
}

func TestRequestHandler_Invalid_param_broadcasterID(t *testing.T) {
	qp := map[string]string{
		"game":           testGameID,
		"broadcaster_id": "",
	}
	badReq := events.APIGatewayProxyRequest{
		HTTPMethod:            "GET",
		QueryStringParameters: qp,
		Resource:              resourceForDetails,
	}
	resp, err := mainPkg.HandleRequest(badReq)
	assert.Equal(t, resp.Body, mainPkg.ErrMsgInvalidBroadcasterID)
	assert.Equal(t, resp.StatusCode, http.StatusBadRequest)
	assert.Nil(t, err)
}

func TestRequestHandler_Invalid_resource(t *testing.T) {
	qp := map[string]string{
		"game": testGameID,
	}
	badReq := events.APIGatewayProxyRequest{
		HTTPMethod:            "GET",
		QueryStringParameters: qp,
		Resource:              "/anything",
	}
	resp, err := mainPkg.HandleRequest(badReq)
	assert.Equal(t, resp.Body, mainPkg.ErrMsgResouceNotFound)
	assert.Equal(t, resp.StatusCode, http.StatusNotFound)
	assert.Nil(t, err)
}

func TestHandleRequestForStreamNames_no_streams(t *testing.T) {
	mockDataService := initTestDataService()
	mockDataService.GetStreamsForGameReturns(nil, nil)
	statusCode, body := mainPkg.HandleRequestForStreamNames(testGameID, mockDataService)
	assert.Equal(t, http.StatusOK, statusCode)
	assert.Nil(t, body)
}

func TestHandleRequestForStreamNames_has_streams(t *testing.T) {
	mockDataService := initTestDataService()
	streamNameArr := []string{"game.1234.1", "game.1234.2", "game.1234.3"}
	mockDataService.GetStreamsForGameReturns(&data.StreamList{
		Value: aws.StringSlice(streamNameArr),
	}, nil)
	statusCode, body := mainPkg.HandleRequestForStreamNames(testGameID, mockDataService)
	gs := &data.StreamList{}
	err := json.Unmarshal([]byte(*body), gs)
	assert.Nil(t, err)
	assert.Equal(t, http.StatusOK, statusCode)
	assert.Equal(t, 3, len(gs.Value))
	assert.Equal(t, streamNameArr[0], *gs.Value[0])
	assert.Equal(t, streamNameArr[1], *gs.Value[1])
	assert.Equal(t, streamNameArr[2], *gs.Value[2])
}

func TestHandleRequestForStreamDetails_no_kinesis_data(t *testing.T) {
	mockDataService := initTestDataService()
	mockDataService.GetShardIDReturns(nil, nil)
	statusCode, body := mainPkg.HandleRequestForStreamsDetails(testGameID, testBroadcasterID, mockDataService)
	assert.Equal(t, mainPkg.ErrMsgShardRecordNotFound, *body)
	assert.Equal(t, http.StatusNotFound, statusCode)
}

func TestHandleRequestForStreamDetails(t *testing.T) {
	mockDataService := initTestDataService()
	mockDataService.GetShardIDReturns(aws.String(testShardID), nil)
	streamNameArr := []string{"game.1234.1", "game.1234.2", "game.1234.3"}
	mockDataService.GetStreamsForGameReturns(&data.StreamList{
		Value: aws.StringSlice(streamNameArr),
	}, nil)
	statusCode, body := mainPkg.HandleRequestForStreamsDetails(testGameID, testBroadcasterID, mockDataService)
	gs := &data.StreamDetails{}
	err := json.Unmarshal([]byte(*body), gs)
	assert.Nil(t, err)
	assert.Equal(t, http.StatusOK, statusCode)
	assert.Equal(t, testShardID, *gs.Shard)
	assert.True(t, strings.HasPrefix(*gs.StreamName, "game.1234."))
}
