package main

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

type dataService struct {
	kinesisClient  *kinesis.Kinesis
	dynamoDBClient *dynamodb.DynamoDB
}

const (
	tableName = "GameStream"
)

func initDynamoDBClient(s *session.Session) *dynamodb.DynamoDB {
	return dynamodb.New(s, &aws.Config{
		Region: aws.String(awsRegion),
	})
}

func initKinesisClient(s *session.Session) *kinesis.Kinesis {
	return kinesis.New(s, &aws.Config{Region: aws.String(awsRegion)})
}

func initDataService() *dataService {
	s := session.Must(session.NewSessionWithOptions(session.Options{
		SharedConfigState: session.SharedConfigEnable,
	}))
	return &dataService{
		kinesisClient:  initKinesisClient(s),
		dynamoDBClient: initDynamoDBClient(s),
	}
}

func (ds *dataService) getStreamsForGame(gameID string) (*GameStream, error) {
	result, err := ds.dynamoDBClient.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(tableName),
		Key: map[string]*dynamodb.AttributeValue{
			"game_id": {
				S: aws.String(gameID),
			},
		},
	})
	if err != nil {
		return nil, err
	}

	gs := GameStream{}
	err = dynamodbattribute.UnmarshalMap(result.Item, &gs)
	if len(gs.MainStreamARN) > 0 {
		return &gs, nil
	}
	return nil, nil
}

func (ds *dataService) saveGameSteamEntity(gs *GameStream) error {
	av, err := dynamodbattribute.MarshalMap(gs)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		Item:      av,
		TableName: aws.String(tableName),
	}

	_, err = ds.dynamoDBClient.PutItem(input)
	if err != nil {
		return err
	}
	return nil
}

func (ds *dataService) updateGameStreamEntity(gs *GameStream) error {
	input := &dynamodb.UpdateItemInput{
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":main": {
				S: aws.String(gs.MainStreamARN),
			},
		},
		TableName: aws.String(tableName),
		Key: map[string]*dynamodb.AttributeValue{
			"game_id": {
				S: aws.String(gs.GameID),
			},
		},
		ReturnValues:     aws.String("ALL_NEW"),
		UpdateExpression: aws.String("set main_stream_arn = :main, clone_stream_arns = :clone"),
	}

	if gs.CloneStreamARNs != nil {
		input.ExpressionAttributeValues[":clone"] = &dynamodb.AttributeValue{SS: aws.StringSlice(*gs.CloneStreamARNs)}
	} else {
		isNull := true
		input.ExpressionAttributeValues[":clone"] = &dynamodb.AttributeValue{NULL: &isNull}
	}

	_, err := ds.dynamoDBClient.UpdateItem(input)
	if err != nil {
		return err
	}
	return nil
}
