package main

import (
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
)

// Max consumers per stream var?
// Number of clones per game var?
// Override to give specific game additional clones?
const (
	delimiter             = "."
	numberOfClonedStreams = 5
)

const (
	errMsgHTTPMethodNotAllowed = "Method Not Allowed"
	errMsgResourceNotFound     = "resource not found"
	errMsgInvalidGameID        = "game_id is invalid"
	errMsgInvalidMainStreamARN = "main_stream_arn is invalid"
	errMsgJSONUnmarshallError  = "cannot unmarshall request body"
	errMsgFromReadingDynamoDB  = "failed to read from dynamoDB"
	errMsgFromWritingDynamoDB  = "failed to write data into dynamoDB"
)

const (
	awsRegion = "us-west-2"
)

func formatStreamName(gameID string, number int) string {
	return fmt.Sprintf("stream.%s.%d", gameID, number)
}

func HandleRequest(request events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	if request.HTTPMethod != "POST" {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusMethodNotAllowed, Body: errMsgResourceNotFound}, nil
	}

	if request.Resource != "/streams/updatedb" {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusNotFound, Body: errMsgHTTPMethodNotAllowed}, nil
	}

	streamToUpdate := &GameStream{}
	err := json.Unmarshal([]byte(request.Body), streamToUpdate)
	if err != nil {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusInternalServerError, Body: errMsgJSONUnmarshallError}, nil
	}

	if len(streamToUpdate.GameID) == 0 {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusBadRequest, Body: errMsgInvalidGameID}, nil
	}

	if len(streamToUpdate.MainStreamARN) == 0 {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusBadRequest, Body: errMsgInvalidMainStreamARN}, nil
	}

	dataService := initDataService()
	var gs *GameStream
	if gs, err = dataService.getStreamsForGame(streamToUpdate.GameID); err != nil {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusInternalServerError, Body: errMsgFromReadingDynamoDB}, nil
	}

	if gs != nil {
		err = dataService.updateGameStreamEntity(streamToUpdate)
	} else {
		err = dataService.saveGameSteamEntity(streamToUpdate)
	}

	if err != nil {
		return events.APIGatewayProxyResponse{StatusCode: http.StatusInternalServerError, Body: errMsgFromWritingDynamoDB}, err
	}

	response := events.APIGatewayProxyResponse{StatusCode: http.StatusNoContent}
	return response, nil
}

func main() {
	lambda.Start(HandleRequest)
}
