package sns

import (
	"encoding/json"
	"errors"
	"time"

	settings "code.justin.tv/devhub/mdaas-discovery-tags-validator/internal/http-settings"

	"code.justin.tv/devhub/mdaas-discovery-tags-validator/internal/awsmocks"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"

	log "github.com/sirupsen/logrus"
)

const (
	eventName = "mdaas-tags"
)

type Message struct {
	Default string `json:"default"`
}

// Publisher publishes an event to an SNS topic, logging any errors that may
// have occurred
//go:generate mockery -name Publisher
type Publisher interface {
	Publish(interface{}, string) error
}

// NewPublisher instantiates a new Publisher
func NewPublisher(c Config) (Publisher, error) {

	httpSettings, err := settings.NewHTTPClientWithSettings(settings.ClientSettings{
		ConnectTimeout:        5 * time.Second,
		ExpectContinueTimeout: 1 * time.Second,
		IdleConnTimeout:       90 * time.Second,
		KeepAlive:             30 * time.Second,
		MaxIdleConns:          0,
		MaxIdleConnsPerHost:   100000,
		ResponseHeaderTimeout: 10 * time.Second,
		TLSHandshakeTimeout:   5 * time.Second,
	})

	if err != nil {
		return nil, err
	}

	sess, err := session.NewSession(&aws.Config{
		HTTPClient: httpSettings,
		Region:     aws.String(c.AWSRegion),
	})

	if err != nil {
		return nil, err
	}

	snsClient := sns.New(sess, aws.NewConfig().WithRegion(c.AWSRegion))

	return &publisherImpl{
		snsClient: snsClient,
		config:    c,
	}, nil
}

type publisherImpl struct {
	snsClient snsiface.SNSAPI
	config    Config
}

// publish attempts to publish the given event to the SNS topic.
// Errors during publish will be reported to Rollbar.
func (p *publisherImpl) Publish(data interface{}, topicArn string) error {

	dataBytes, err := json.Marshal(data)
	if err != nil {
		return errors.New("Error marshalling SNS payload")
	}

	message := Message{
		Default: string(dataBytes),
	}

	messageBytes, _ := json.Marshal(message)
	in := &sns.PublishInput{
		Message:          aws.String(string(messageBytes)),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": &sns.MessageAttributeValue{
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(topicArn),
	}

	for i := 0; i < p.config.retryCount(); i++ {
		_, err = p.snsClient.Publish(in)
		if err == nil {
			return nil
		}

		//report retry
		log.Error("Publish retry", err)

		time.Sleep(p.config.retryDelay())
	}
	return err
}

func NewTestPublisher(snsmock *awsmocks.SNSAPI, c *Config) (Publisher, error) {
	return &publisherImpl{snsClient: snsmock}, nil
}
