package sns

import (
	"errors"
	"testing"
	"time"

	"code.justin.tv/devhub/mdaas-discovery-tags-validator/internal/awsmocks"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/stretchr/testify/suite"
)

const (
	arn = "arn:aws:sns:us-east-2:797743463538:dev-mdaas-tags"
)

type PublisherTest struct {
	suite.Suite
	publisher  Publisher
	mockSNSAPI *awsmocks.SNSAPI
}

func (suite *PublisherTest) SetupTest() {
	mSNS := &awsmocks.SNSAPI{}
	p, _ := NewTestPublisher(mSNS, &Config{
		AWSRegion: "us-west-2",
	})
	suite.publisher = p
	suite.mockSNSAPI = mSNS
}

func (suite *PublisherTest) TearDownTest() {
	suite.mockSNSAPI.AssertExpectations(suite.T())
}

func TestRunPublisherSuite(t *testing.T) {
	suite.Run(t, new(PublisherTest))
}

func (suite *PublisherTest) TestPublishSuccessTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(arn),
	}).Return(nil, nil).Once()

	err := suite.publisher.Publish(message, arn)
	suite.NoError(err)
}

func (suite *PublisherTest) TestPublishFailureTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(arn),
	}).Return(nil, errors.New("failed")).Times(10)

	err := suite.publisher.Publish(message, arn)
	suite.Error(err)
}

func (suite *PublisherTest) TestPublishFailureThenSuccessTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(arn),
	}).Return(nil, errors.New("failed sns publish")).
		Times(4).
		Return(nil, nil).
		Times(1)

	err := suite.publisher.Publish(message, arn)
	suite.NoError(err)

	time.Sleep(1 * time.Second)
}
