package processor

import (
	"fmt"

	"code.justin.tv/devhub/mdaas-discovery-tags-validator/internal/clients/sns"
	models "code.justin.tv/devhub/mdaas-discovery-tags-validator/models"
	log "github.com/sirupsen/logrus"
)

var gearBoxTagUUIDs = []string{
	"f9029b6f-a894-482d-a77a-84598859b2a3", // Vault Hunter: Amara
	"d902fd55-1c9d-40ff-bbda-2efe8608f487", // Vault Hunter: FL4K
	"e2fc3838-3c58-44db-8b1c-b5d058acf479", // Vault Hunter: Moze
	"060fbf32-f5b9-4262-8071-3fba1307fa36", // Vault Hunter: Zane
	"a0bbbfd5-a59e-4005-8bcb-22e9c932a15b", // Mode: True Vault Hunter
	"4381920b-dd0c-4696-aad3-41f49202ff57", // Mode: Mayhem
	"79c1e315-d8c5-48b4-af51-04209ca12ee3", // Mode: Mayhem1
	"f62f4792-558e-4002-97f8-0a9ebc5a2a09", // Mode: Mayhem2
	"2bdc21f7-84e4-4bca-95a6-7405e6e33d09", // Mode: Mayhem3
	"ea31603d-9359-4988-bb65-4b7ba3c11564", // Mode: Mayhem4
	"ea4b8406-eeec-4e25-abae-e96243695d9b", // Mode: ProvingGroundsTag
	"11374ec6-05a6-4611-b8db-9ef0286d4edf", // Mode: CircleOfSlaughterTag
	"063b8a73-04fc-46c4-86a2-91289d04c195", // Mode: TakedownTag
	"2d4c0932-083d-4d39-9475-8a545e6202c7", // Mode: NewDLCTag
	"cefd3dbb-3baa-4e51-bfed-e6d5155a4e18", // Mode: Gear Up
}

//go:generate mockery -name Processor
type Processor interface {
	Process(gameData models.GameData)
}

type ProcessorImpl struct {
	devTagsTopicArn  string
	prodTagsTopicArn string
	snsClient        sns.Publisher
}

func NewProcessor(snsClient sns.Publisher, prodTagsTopicArn, devTagsTopicArn string) *ProcessorImpl {
	return &ProcessorImpl{
		devTagsTopicArn,
		prodTagsTopicArn,
		snsClient,
	}
}

func (p ProcessorImpl) Process(gameData models.GameData) {

	topicArn := p.devTagsTopicArn
	if gameData.Env == "prod" {
		topicArn = p.prodTagsTopicArn
	}

	discoveryMessage := models.DiscoveryMessage{}

	for _, tag := range gameData.MetadataDelta.Tags.Added {
		if validateTag(tag) {
			discoveryMessage.Add = append(discoveryMessage.Add, tag)
		} else {
			fmt.Printf("Tag UUID [ %s ] was not found in whitelist", tag)
			// save in dynamo db
		}
	}

	tagsToRemove := gameData.MetadataDelta.Tags.Removed
	if gameData.MetadataDelta.Refresh {
		// remove all un-added tags if refresh is set to true
		tagsToRemove = difference(gearBoxTagUUIDs, gameData.MetadataDelta.Tags.Added)
	}

	for _, tag := range tagsToRemove {
		if validateTag(tag) {
			discoveryMessage.Remove = append(discoveryMessage.Remove, tag)
		} else {
			fmt.Printf("Tag UUID [ %s ] was not found in whitelist", tag)
			// save in dynamo db
		}
	}

	for _, broadcasterID := range gameData.BroadcasterIDs {

		discoveryMessage.ChannelID = broadcasterID

		if err := p.snsClient.Publish(discoveryMessage, topicArn); err != nil {
			log.Error(err)
		}
	}
}

//	Validate tag checks to see if uuid in parameter exists
func validateTag(uuid string) bool {
	for _, gearBoxTagUUID := range gearBoxTagUUIDs {
		if uuid == gearBoxTagUUID {
			return true
		}
	}
	return false
}

// difference returns the an array of strings from the first array of strings which are missing from the second
// array of strings
func difference(a, b []string) (diff []string) {
	m := make(map[string]bool)

	for _, item := range b {
		m[item] = true
	}

	for _, item := range a {
		if _, ok := m[item]; !ok {
			diff = append(diff, item)
		}

	}
	return
}
