package processor

import (
	"testing"

	"code.justin.tv/devhub/mdaas-discovery-tags-validator/internal/clients/sns/mocks"
	models "code.justin.tv/devhub/mdaas-discovery-tags-validator/models"

	"github.com/stretchr/testify/suite"
)

const (
	devTagsTopicArn  = "devArn"
	prodTagsTopicArn = "prodArn"
)

type ProcessorTest struct {
	suite.Suite
	processor    Processor
	snsPublisher *mocks.Publisher
}

func (suite *ProcessorTest) SetupTest() {
	mSNS := &mocks.Publisher{}
	suite.snsPublisher = mSNS
	suite.processor = NewProcessor(mSNS, prodTagsTopicArn, devTagsTopicArn)
}

func (suite *ProcessorTest) TearDownTest() {
	suite.snsPublisher.AssertExpectations(suite.T())
}

func TestRunPublisherSuite(t *testing.T) {
	suite.Run(t, new(ProcessorTest))
}

// Successfully publishes to prod
func (suite *ProcessorTest) TestProcessSuccess() {
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add:    []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
			Remove: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
				Removed: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
			},
		},
	})
}

// Successfully publishes to dev
func (suite *ProcessorTest) TestProcessDevSuccess() {
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add:    []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
			Remove: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
		},
		prodTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "prod",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
				Removed: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
			},
		},
	})
}

// Successfully publishes to multiple channels
func (suite *ProcessorTest) TestProcessMultipleChannelsSuccess() {
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add:    []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
			Remove: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel2",
			Add:    []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
			Remove: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1", "testChannel2"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
				Removed: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
			},
		},
	})
}

// Successfully filters out un whitelisted tags
func (suite *ProcessorTest) TestProcessFiltersUnwhitelistedTags() {
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add:    []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487"},
			Remove: []string{"4381920b-dd0c-4696-aad3-41f49202ff57"},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "d902fd55-1c9d-40ff-bbda-2efe8608f487", "UnwhitelistedTag1"},
				Removed: []string{"4381920b-dd0c-4696-aad3-41f49202ff57", "UnwhitelistedTag2"},
			},
		},
	})
}

// Successfully filters out un whitelisted tags
func (suite *ProcessorTest) TestProcessHandlesMessagesWithNoTags() {
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add:    []string(nil),
			Remove: []string(nil),
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string(nil),
				Removed: []string(nil),
			},
		},
	})
}

// Removes old tags when adding new tags when refresh is set to true
func (suite *ProcessorTest) TestProcessHandlesMessageWithRefresh() {

	// test refresh when no tags are added or remove
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add: []string(nil),
			Remove: []string{
				"f9029b6f-a894-482d-a77a-84598859b2a3",
				"d902fd55-1c9d-40ff-bbda-2efe8608f487",
				"e2fc3838-3c58-44db-8b1c-b5d058acf479",
				"060fbf32-f5b9-4262-8071-3fba1307fa36",
				"a0bbbfd5-a59e-4005-8bcb-22e9c932a15b",
				"4381920b-dd0c-4696-aad3-41f49202ff57",
				"79c1e315-d8c5-48b4-af51-04209ca12ee3",
				"f62f4792-558e-4002-97f8-0a9ebc5a2a09",
				"2bdc21f7-84e4-4bca-95a6-7405e6e33d09",
				"ea31603d-9359-4988-bb65-4b7ba3c11564",
				"ea4b8406-eeec-4e25-abae-e96243695d9b",
				"11374ec6-05a6-4611-b8db-9ef0286d4edf",
				"063b8a73-04fc-46c4-86a2-91289d04c195",
				"2d4c0932-083d-4d39-9475-8a545e6202c7",
				"cefd3dbb-3baa-4e51-bfed-e6d5155a4e18",
			},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string(nil),
				Removed: []string(nil),
			},
			Refresh: true,
		},
	})

	// test when some tags are added
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add: []string{
				"f9029b6f-a894-482d-a77a-84598859b2a3",
				"a0bbbfd5-a59e-4005-8bcb-22e9c932a15b",
			},
			Remove: []string{
				"d902fd55-1c9d-40ff-bbda-2efe8608f487",
				"e2fc3838-3c58-44db-8b1c-b5d058acf479",
				"060fbf32-f5b9-4262-8071-3fba1307fa36",
				"4381920b-dd0c-4696-aad3-41f49202ff57",
				"79c1e315-d8c5-48b4-af51-04209ca12ee3",
				"f62f4792-558e-4002-97f8-0a9ebc5a2a09",
				"2bdc21f7-84e4-4bca-95a6-7405e6e33d09",
				"ea31603d-9359-4988-bb65-4b7ba3c11564",
				"ea4b8406-eeec-4e25-abae-e96243695d9b",
				"11374ec6-05a6-4611-b8db-9ef0286d4edf",
				"063b8a73-04fc-46c4-86a2-91289d04c195",
				"2d4c0932-083d-4d39-9475-8a545e6202c7",
				"cefd3dbb-3baa-4e51-bfed-e6d5155a4e18",
			},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "a0bbbfd5-a59e-4005-8bcb-22e9c932a15b"},
				Removed: []string(nil),
			},
			Refresh: true,
		},
	})

	// test refresh when some tags are remove
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add: []string(nil),
			Remove: []string{
				"f9029b6f-a894-482d-a77a-84598859b2a3",
				"d902fd55-1c9d-40ff-bbda-2efe8608f487",
				"e2fc3838-3c58-44db-8b1c-b5d058acf479",
				"060fbf32-f5b9-4262-8071-3fba1307fa36",
				"a0bbbfd5-a59e-4005-8bcb-22e9c932a15b",
				"4381920b-dd0c-4696-aad3-41f49202ff57",
				"79c1e315-d8c5-48b4-af51-04209ca12ee3",
				"f62f4792-558e-4002-97f8-0a9ebc5a2a09",
				"2bdc21f7-84e4-4bca-95a6-7405e6e33d09",
				"ea31603d-9359-4988-bb65-4b7ba3c11564",
				"ea4b8406-eeec-4e25-abae-e96243695d9b",
				"11374ec6-05a6-4611-b8db-9ef0286d4edf",
				"063b8a73-04fc-46c4-86a2-91289d04c195",
				"2d4c0932-083d-4d39-9475-8a545e6202c7",
				"cefd3dbb-3baa-4e51-bfed-e6d5155a4e18",
			},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string(nil),
				Removed: []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "a0bbbfd5-a59e-4005-8bcb-22e9c932a15b"},
			},
			Refresh: true,
		},
	})

	// test refresh when tags are added and removed
	suite.snsPublisher.On("Publish",
		models.DiscoveryMessage{ChannelID: "testChannel1",
			Add: []string{
				"060fbf32-f5b9-4262-8071-3fba1307fa36",
			},
			Remove: []string{
				"f9029b6f-a894-482d-a77a-84598859b2a3",
				"d902fd55-1c9d-40ff-bbda-2efe8608f487",
				"e2fc3838-3c58-44db-8b1c-b5d058acf479",
				"a0bbbfd5-a59e-4005-8bcb-22e9c932a15b",
				"4381920b-dd0c-4696-aad3-41f49202ff57",
				"79c1e315-d8c5-48b4-af51-04209ca12ee3",
				"f62f4792-558e-4002-97f8-0a9ebc5a2a09",
				"2bdc21f7-84e4-4bca-95a6-7405e6e33d09",
				"ea31603d-9359-4988-bb65-4b7ba3c11564",
				"ea4b8406-eeec-4e25-abae-e96243695d9b",
				"11374ec6-05a6-4611-b8db-9ef0286d4edf",
				"063b8a73-04fc-46c4-86a2-91289d04c195",
				"2d4c0932-083d-4d39-9475-8a545e6202c7",
				"cefd3dbb-3baa-4e51-bfed-e6d5155a4e18",
			},
		},
		devTagsTopicArn,
	).Return(nil).Once()

	suite.processor.Process(models.GameData{
		BroadcasterIDs: []string{"testChannel1"},
		Env:            "dev",
		GameID:         "12345",
		MetadataDelta: models.MetadataDelta{
			Active: true,
			ID:     "12345679",
			Tags: models.TagsDelta{
				Added:   []string{"060fbf32-f5b9-4262-8071-3fba1307fa36"},
				Removed: []string{"f9029b6f-a894-482d-a77a-84598859b2a3", "a0bbbfd5-a59e-4005-8bcb-22e9c932a15b"},
			},
			Refresh: true,
		},
	})
}

func (suite *ProcessorTest) TestDifference() {
	stringListA := []string{
		"abcd",
		"1234",
	}

	stringListSameAsA := []string{
		"abcd",
		"1234",
	}

	stringListDifferent := []string{
		"test",
		"anotherthing",
		"abcd",
		"1234",
	}

	// 2 string lists with equal values return an empty array
	suite.EqualValues([]string(nil), difference(stringListA, stringListSameAsA))

	// 2 string lists with unequal values returns values missing from first array
	suite.EqualValues([]string{"test", "anotherthing"}, difference(stringListDifferent, stringListA))

	// Returns nil string array if 1st array is a subset of the second array
	suite.EqualValues([]string(nil), difference(stringListA, stringListDifferent))
}
