import { App } from '@aws-cdk/core';
import {
  ALL_SNS_ACCESS_ACCOUNT_ID_DEV,
  DEV_ACCOUNT_ID,
  E2_INGEST_HTTP_ACCOUNT_ID_DEV,
  GRAFFITI_ACCOUNT_ID,
  LEAGUE_CME_ACCOUNT_ID_DEV,
  PRIMARY_REGION,
} from './consts';
import { DevZonesStack } from './stacks/dev-zones';
import { DynamoAllowlistStack } from './stacks/dynamo-allowlist';
import { IngestCertsStack } from './stacks/ingest-certs';
import { IngestDebugLogsStack } from './stacks/ingest-debug-logs';
import { IngestReposStack } from './stacks/ingest-repos';
import { IngestSecretsStack } from './stacks/ingest-secrets';
import { IngestServiceStack } from './stacks/ingest-service';
import { JenkinsDeploysStack } from './stacks/jenkins-deploys';
import { SnsTopicsStack } from './stacks/sns-topics';
import { TagsValidatorStack } from './stacks/tags-validator';
import { TagsValidatorBucketsStack } from './stacks/tags-validator-buckets';
import { VpcStack } from './stacks/vpc';

const env = {
  account: DEV_ACCOUNT_ID,
  region: PRIMARY_REGION,
};

export function createDevStacks(app: App) {
  // Common dev stacks
  const vpcStack = new VpcStack(app, 'DevVpc', {
    env,
    vpcId: 'vpc-00bb9e143bf9366c8',
  });
  const devZonesStack = new DevZonesStack(app, { env });

  // SNS Topics
  const snsTopicStack = new SnsTopicsStack(app, 'DevSnsTopics', {
    env,
    graffitiAccountID: GRAFFITI_ACCOUNT_ID,
    leagueCMEAccountIDs: [LEAGUE_CME_ACCOUNT_ID_DEV],
    allAccessAccountIDs: ALL_SNS_ACCESS_ACCOUNT_ID_DEV,
    e2IngestHttpAccountID: E2_INGEST_HTTP_ACCOUNT_ID_DEV,
  });

  // DynamoDB Allowlist
  const dynamoAllowlistStack = new DynamoAllowlistStack(app, 'DevDynamoAllowlist', {
    env,
    e2IngestHttpAccountID: E2_INGEST_HTTP_ACCOUNT_ID_DEV,
  });

  // Ingest Service
  const ingestDomainName = 'ingest.dev.metadata.twitch.a2z.com';
  const ingestSecretsStack = new IngestSecretsStack(app, 'DevIngestSecret', { env });
  const ingestReposStack = new IngestReposStack(app, 'DevIngestRepo', { env });
  const ingestCertsStack = new IngestCertsStack(app, 'DevIngestCert', {
    env,
    domainName: ingestDomainName,
    hostedZone: devZonesStack.internalZone,
  });
  const ingestDebugLogsStack = new IngestDebugLogsStack(app, 'DevIngestDebugLogs', { env });
  const ingestServiceStack = new IngestServiceStack(app, 'DevIngestService', {
    env,
    certificate: ingestCertsStack.primary,
    domainName: ingestDomainName,
    hostedZone: devZonesStack.internalZone,
    alertsTopic: snsTopicStack.alertsTopic,
    repository: ingestReposStack.primary,
    secret: ingestSecretsStack.primary,
    vpc: vpcStack.vpc,
    debugLogsFirehose: ingestDebugLogsStack.firehose,
  });

  // Tags Validator Bucket
  const tagsValidatorBucketStack = new TagsValidatorBucketsStack(app, 'DevTagsValidatorBucket', { env });

  // Tags Validator Lambda
  const tagsValidatorStack = new TagsValidatorStack(app, 'DevTagsValidator', {
    env,
    tagsDevTopic: snsTopicStack.devTagsTopic,
    tagsProdTopic: snsTopicStack.prodTagsTopic,
    metadataTopic: snsTopicStack.devMetadataTopic,
    alertsTopic: snsTopicStack.alertsTopic,
    bucket: tagsValidatorBucketStack.bucket,
  });

  // Give Jenkins access to deploy
  new JenkinsDeploysStack(app, 'DevJenkinsDeploys', {
    env,
    lambdaFunctions: [
      {
        bucket: tagsValidatorBucketStack.bucket,
        lambdaFunction: tagsValidatorStack.tagsValidatorLambda,
      },
    ],
    services: [
      {
        service: ingestServiceStack.service,
        repo: ingestReposStack.primary,
        secret: ingestSecretsStack.primary,
      },
      {
        service: ingestServiceStack.canary,
        repo: ingestReposStack.primary,
      },
    ],
    allowlist: dynamoAllowlistStack.dynamoTable,
  });
}
