import { App } from '@aws-cdk/core';
import {
  ALL_SNS_ACCESS_ACCOUNT_ID_PROD,
  E2_INGEST_HTTP_ACCOUNT_ID_PROD,
  GRAFFITI_ACCOUNT_ID,
  LEAGUE_CME_ACCOUNT_ID_DEV,
  LEAGUE_CME_ACCOUNT_ID_PROD,
  PAGER_DUTY_URL,
  PRIMARY_REGION,
  PROD_ACCOUNT_ID,
} from './consts';
import { DynamoAllowlistStack } from './stacks/dynamo-allowlist';
import { IngestCertsStack } from './stacks/ingest-certs';
import { IngestDebugLogsStack } from './stacks/ingest-debug-logs';
import { IngestReposStack } from './stacks/ingest-repos';
import { IngestSecretsStack } from './stacks/ingest-secrets';
import { IngestServiceStack } from './stacks/ingest-service';
import { JenkinsDeploysStack } from './stacks/jenkins-deploys';
import { ProdZonesStack } from './stacks/prod-zones';
import { SnsTopicsStack } from './stacks/sns-topics';
import { TagsValidatorStack } from './stacks/tags-validator';
import { TagsValidatorBucketsStack } from './stacks/tags-validator-buckets';
import { VpcStack } from './stacks/vpc';

const env = {
  account: PROD_ACCOUNT_ID,
  region: PRIMARY_REGION,
};

export function createProdStacks(app: App) {
  // Common prod stacks
  const vpcStack = new VpcStack(app, 'ProdVpc', {
    env,
    vpcId: 'vpc-0474141c268f9e633',
  });
  const zonesStack = new ProdZonesStack(app, { env });

  // SNS Topics
  const snsTopicStack = new SnsTopicsStack(app, 'ProdSnsTopics', {
    env,
    graffitiAccountID: GRAFFITI_ACCOUNT_ID,
    leagueCMEAccountIDs: [LEAGUE_CME_ACCOUNT_ID_PROD, LEAGUE_CME_ACCOUNT_ID_DEV],
    allAccessAccountIDs: ALL_SNS_ACCESS_ACCOUNT_ID_PROD,
    pagerDutyURL: PAGER_DUTY_URL,
    e2IngestHttpAccountID: E2_INGEST_HTTP_ACCOUNT_ID_PROD,
  });

  // DynamoDB Allowlist
  new DynamoAllowlistStack(app, 'ProdDynamoAllowlist', {
    env,
    e2IngestHttpAccountID: E2_INGEST_HTTP_ACCOUNT_ID_PROD,
  });

  // Ingest Service
  const ingestDomainName = 'metadata.twitch.tv';
  const ingestSecretsStack = new IngestSecretsStack(app, 'ProdIngestSecret', { env });
  const ingestReposStack = new IngestReposStack(app, 'ProdIngestRepo', { env });
  const ingestCertsStack = new IngestCertsStack(app, 'ProdIngestCert', {
    domainName: ingestDomainName,
    env,
    hostedZone: zonesStack.publicZone,
  });
  const ingestDebugLogsStack = new IngestDebugLogsStack(app, 'ProdIngestDebugLogs', { env });
  const ingestServiceStack = new IngestServiceStack(app, 'ProdIngestService', {
    certificate: ingestCertsStack.primary,
    domainName: ingestDomainName,
    env,
    hostedZone: zonesStack.publicZone,
    alertsTopic: snsTopicStack.alertsTopic,
    repository: ingestReposStack.primary,
    secret: ingestSecretsStack.primary,
    vpc: vpcStack.vpc,
    prod: true,
    debugLogsFirehose: ingestDebugLogsStack.firehose,
  });

  // Tags Validator Bucket
  const tagsValidatorBucketStack = new TagsValidatorBucketsStack(app, 'ProdTagsValidatorBucket', { env });

  // Tags Validator Lambda
  const tagsValidatorStack = new TagsValidatorStack(app, 'ProdTagsValidator', {
    env,
    tagsDevTopic: snsTopicStack.devTagsTopic,
    tagsProdTopic: snsTopicStack.prodTagsTopic,
    metadataTopic: snsTopicStack.devMetadataTopic,
    alertsTopic: snsTopicStack.alertsTopic,
    bucket: tagsValidatorBucketStack.bucket,
  });

  // Give Jenkins access to deploy
  new JenkinsDeploysStack(app, 'ProdJenkinsDeploys', {
    env,
    lambdaFunctions: [
      {
        bucket: tagsValidatorBucketStack.bucket,
        lambdaFunction: tagsValidatorStack.tagsValidatorLambda,
      },
    ],
    services: [
      {
        service: ingestServiceStack.service,
        repo: ingestReposStack.primary,
      },
      {
        service: ingestServiceStack.canary,
        repo: ingestReposStack.primary,
      },
    ],
  });
}
