import { IRepository } from '@aws-cdk/aws-ecr';
import { AwsLogDriver, ContainerImage, FargateTaskDefinition, TaskDefinition, UlimitName } from '@aws-cdk/aws-ecs';
import { Effect, ManagedPolicy, PolicyStatement } from '@aws-cdk/aws-iam';
import { RetentionDays } from '@aws-cdk/aws-logs';
import { Construct } from '@aws-cdk/core';

interface IngestServiceTaskProps {
  /** If true, service will be publicly accesible, and scaled to production levels. */
  prod?: boolean;
  /** If true, service will have env var canary to be set true. */
  canary?: boolean;
  /** ECR repository containing the service. */
  repository: IRepository;
  /** Service secrets Policy */
  secretsManagerPolicy: PolicyStatement;
  /** Firehose delivery stream Policy to deliver debug logs */
  firehosePolicy: PolicyStatement;
}

export class IngestServiceTask extends Construct {
  public readonly taskDef: TaskDefinition;

  constructor(scope: Construct, name: string, props: IngestServiceTaskProps) {
    super(scope, name);

    // ECS Task Definition
    const taskDefinition = new FargateTaskDefinition(this, 'Def', {
      cpu: 1024,
      memoryLimitMiB: 4096,
    });

    taskDefinition.addToTaskRolePolicy(props.secretsManagerPolicy);
    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );
    // Permissions to publish to all SNS topics in the account
    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['sns:Publish'],
        resources: ['*'],
      })
    );
    taskDefinition.addToTaskRolePolicy(props.firehosePolicy);
    // policy used by S2S client: https://wiki.twitch.com/pages/viewpage.action?pageId=183990900
    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['sts:AssumeRole'],
        resources: ['arn:aws:iam::180116294062:role/malachai/*'],
      })
    );
    // Full access to DynamoDB
    taskDefinition.taskRole.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName('AmazonDynamoDBFullAccess'));

    const streamPrefix = `IngestService${props.canary ? 'Canary' : ''}`;

    // ECS Container
    const container = taskDefinition.addContainer('IngestService', {
      image: ContainerImage.fromEcrRepository(props.repository, 'latest'),
      logging: new AwsLogDriver({ streamPrefix, logRetention: RetentionDays.THREE_MONTHS }),
      environment: {
        ENV: props.prod ? 'production' : 'staging',
        CANARY: props.canary ? 'true' : 'false',
      },
    });
    container.addPortMappings({ containerPort: 80 });
    container.addUlimits({
      name: UlimitName.NOFILE,
      softLimit: 32768,
      hardLimit: 32768,
    });

    this.taskDef = taskDefinition;
  }
}
