import { AttributeType, BillingMode, ProjectionType, Table } from '@aws-cdk/aws-dynamodb';
import { AccountPrincipal, Role } from '@aws-cdk/aws-iam';
import { Construct, Stack, StackProps } from '@aws-cdk/core';

interface DynamoAllowlistStackProps extends StackProps {
  e2IngestHttpAccountID: string;
}

export class DynamoAllowlistStack extends Stack {
  public readonly dynamoTable: Table;

  constructor(scope: Construct, name: string, props: DynamoAllowlistStackProps) {
    super(scope, name, props);

    const table = new Table(this, 'DynamoDBTable', {
      partitionKey: {
        name: 'GameID',
        type: AttributeType.STRING,
      },
      sortKey: {
        name: 'ClientID',
        type: AttributeType.STRING,
      },
      billingMode: BillingMode.PAY_PER_REQUEST,
      pointInTimeRecovery: true,
    });

    const indexNames = ['ClientID', 'OrganizationID'];
    indexNames.map(indexName => {
      table.addGlobalSecondaryIndex({
        indexName: `${indexName}Index`,
        partitionKey: {
          name: `${indexName}`,
          type: AttributeType.STRING,
        },
        projectionType: ProjectionType.ALL,
      });
    });

    this.dynamoTable = table;

    const role = new Role(this, 'IAMRole', {
      assumedBy: new AccountPrincipal(props.e2IngestHttpAccountID),
    });
    table.grantReadWriteData(role.grantPrincipal);
  }
}
