import { Effect, IRole, PolicyStatement, Role, ServicePrincipal } from '@aws-cdk/aws-iam';
import { CfnDeliveryStream } from '@aws-cdk/aws-kinesisfirehose';
import { Bucket, IBucket } from '@aws-cdk/aws-s3';
import { Construct, Duration, Stack, StackProps } from '@aws-cdk/core';

// Respect to GDPR. Storing the data less than 28 days.
const DEFAULT_LOGS_EXPIRATION_IN_DAYS = Duration.days(20);

/**
 *  use this stack to create new firehose delivery stream for mdaas-ingest
 */
export class IngestDebugLogsStack extends Stack {
  public readonly firehose: CfnDeliveryStream;
  private firehoseRole: IRole;
  private bucket: IBucket;

  constructor(scope: Construct, name: string, props: StackProps) {
    super(scope, name, props);

    // these 3 calls need to be called in order
    this.bucket = this.initBucket();
    this.firehoseRole = this.initFirehoseRole();
    this.firehose = this.initFirehose();
  }

  private initBucket(): IBucket {
    return new Bucket(this, 'debugLogs', {
      lifecycleRules: [
        {
          expiration: DEFAULT_LOGS_EXPIRATION_IN_DAYS,
        },
      ],
    });
  }

  private initFirehoseRole(): IRole {
    const role = new Role(this, 'debugLogsFirehoseRole', {
      assumedBy: new ServicePrincipal('firehose.amazonaws.com'),
    });
    role.addToPolicy(
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: [
          's3:AbortMultipartUpload',
          's3:GetBucketLocation',
          's3:GetObject',
          's3:ListBucket',
          's3:ListBucketMultipartUploads',
          's3:PutObject',
        ],
        resources: [this.bucket.bucketArn, `${this.bucket.bucketArn}/*`],
      })
    );
    return role;
  }

  private initFirehose(): CfnDeliveryStream {
    return new CfnDeliveryStream(this, 'debugLogsFirehose', {
      deliveryStreamType: 'DirectPut',
      s3DestinationConfiguration: {
        bucketArn: this.bucket.bucketArn,
        bufferingHints: {
          intervalInSeconds: 60 * 5, // 5 min
          sizeInMBs: 128,
        },
        compressionFormat: 'UNCOMPRESSED',
        roleArn: this.firehoseRole.roleArn,
      },
    });
  }
}
