import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { Alarm, ComparisonOperator, Metric } from '@aws-cdk/aws-cloudwatch';
import { IVpc, Peer, Port, SecurityGroup } from '@aws-cdk/aws-ec2';
import { IRepository } from '@aws-cdk/aws-ecr';
import { Cluster, FargatePlatformVersion, FargateService } from '@aws-cdk/aws-ecs';
import { ApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';
import { Effect, PolicyStatement } from '@aws-cdk/aws-iam';
import { CfnDeliveryStream } from '@aws-cdk/aws-kinesisfirehose';
import { AddressRecordTarget, ARecord, IHostedZone } from '@aws-cdk/aws-route53';
import { LoadBalancerTarget } from '@aws-cdk/aws-route53-targets';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { ITopic } from '@aws-cdk/aws-sns';
import { Construct, Duration, Stack, StackProps, Tag } from '@aws-cdk/core';
import { DEV_ACCOUNT_ID } from '../consts';
import { IngestServiceTask } from '../resources/ingest-service-task';

interface IngestServiceStackProps extends StackProps {
  vpc: IVpc;
  /** If true, service will be publicly accesible, and scaled to production levels. */
  prod?: boolean;
  /** Fully qualified Route 53 record name to host the service on. Leave empty to use the zone root record */
  domainName: string;
  /** SNS Topic for alerts */
  alertsTopic: ITopic;
  /** Route 53 zone to host the service on. */
  hostedZone: IHostedZone;
  /** ACM Certificate to use. */
  certificate: ICertificate;
  /** ECR repository containing the service. */
  repository: IRepository;
  /** Service secrets */
  secret: ISecret;
  /** Firehose delivery stream to deliver debug logs */
  debugLogsFirehose: CfnDeliveryStream;
}

/**
 * Ingest Service ECS cluster, service, load balancer, and DNS record.
 */
export class IngestServiceStack extends Stack {
  public readonly service: FargateService;
  public readonly canary: FargateService;

  constructor(scope: Construct, name: string, props: IngestServiceStackProps) {
    super(scope, name, props);

    // ECS Cluster
    const cluster = new Cluster(this, 'Cluster', {
      vpc: props.vpc,
    });
    Tag.add(cluster, 'DeployCluster', 'Ingest');

    const secretsManagerPolicy = new PolicyStatement({
      effect: Effect.ALLOW,
      actions: ['secretsmanager:GetSecretValue'],
      resources: [props.secret.secretArn],
    });

    const firehosePolicy = new PolicyStatement({
      effect: Effect.ALLOW,
      actions: ['firehose:PutRecord', 'firehose:PutRecordBatch'],
      resources: [props.debugLogsFirehose.attrArn],
    });

    // ECS Task Definition
    const taskDefinition = new IngestServiceTask(this, 'Task', {
      prod: props.prod,
      repository: props.repository,
      firehosePolicy,
      secretsManagerPolicy,
    }).taskDef;

    // ECS Canary Task Definition
    const taskDefinitionCanary = new IngestServiceTask(this, 'CanaryTask', {
      prod: props.prod,
      canary: true,
      repository: props.repository,
      firehosePolicy,
      secretsManagerPolicy,
    }).taskDef;

    // ECS Service
    this.service = new FargateService(this, 'Service', {
      cluster,
      desiredCount: props.prod ? 3 : 1,
      taskDefinition,
      platformVersion: FargatePlatformVersion.VERSION1_4,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
    });

    // ECS Service Auto Scaling
    const scaling = this.service.autoScaleTaskCount({
      minCapacity: props.prod ? 3 : 1,
      maxCapacity: props.prod ? 30 : 3,
    });

    scaling.scaleOnCpuUtilization('Cpu', {
      targetUtilizationPercent: 60,
    });

    scaling.scaleOnMemoryUtilization('Memory', {
      targetUtilizationPercent: 80,
    });

    // Canary Service
    this.canary = new FargateService(this, 'CanaryService', {
      cluster,
      desiredCount: 1,
      taskDefinition: taskDefinitionCanary,
      platformVersion: FargatePlatformVersion.VERSION1_4,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
    });

    // Security Group -- Restricted to WPA2 using Amazon Prefix Lists (https://apll.corp.amazon.com/) when prod is false.
    const securityGroup = new SecurityGroup(this, 'LoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    securityGroup.addIngressRule(props.prod ? Peer.anyIpv4() : Peer.prefixList('pl-f8a64391'), Port.tcp(443));

    // For dev account, add NAT gateway IPs to allow load testing
    if (this.account === DEV_ACCOUNT_ID) {
      securityGroup.addIngressRule(Peer.ipv4('34.211.0.142/32'), Port.tcp(443), 'NAT Gateway');
      securityGroup.addIngressRule(Peer.ipv4('34.215.139.247/32'), Port.tcp(443), 'NAT Gateway');
      securityGroup.addIngressRule(Peer.ipv4('52.10.216.75/32'), Port.tcp(443), 'NAT Gateway');
    }

    // Load Balancer
    const loadBalancer = new ApplicationLoadBalancer(this, 'LoadBalancer', {
      vpc: props.vpc,
      internetFacing: true,
      securityGroup,
      idleTimeout: Duration.seconds(180),
    });

    // Load Balancer Listener
    const listener = loadBalancer.addListener('Listener', {
      port: 443,
      certificateArns: [props.certificate.certificateArn],
      open: false,
    });

    listener.addFixedResponse('Default', {
      statusCode: '404',
    });

    // Load Balancer Target Group
    const targetGroup = listener.addTargets('Service', {
      pathPattern: '/api/ingest',
      priority: 1,
      port: 80,
      deregistrationDelay: Duration.seconds(60),
      healthCheck: {
        path: '/ping',
        healthyHttpCodes: '200',
      },
    });
    targetGroup.addTarget(this.service);
    targetGroup.addTarget(this.canary);

    // DNS Record
    new ARecord(this, 'DnsRecord', {
      zone: props.hostedZone,
      recordName: props.domainName,
      target: AddressRecordTarget.fromAlias(new LoadBalancerTarget(loadBalancer)),
    });

    // Add alarm for zero connections
    const liveConnectionsMetric = new Metric({
      namespace: 'ingest',
      metricName: 'LiveConnections',
      dimensions: {
        Region: 'us-west-2',
        Service: 'ingest',
        Stage: 'production',
      },
    });

    const connectionsAlarm = new Alarm(this, 'ZeroConnectionsAlarm', {
      alarmDescription: 'Notify pagerduty if the number of connections is 0 for 1 period.',
      evaluationPeriods: 5,
      metric: liveConnectionsMetric,
      period: Duration.minutes(3),
      threshold: 0,
      comparisonOperator: ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
    });

    connectionsAlarm.addAlarmAction({
      bind: () => ({ alarmActionArn: props.alertsTopic.topicArn }),
    });
  }
}
