import { ITable } from '@aws-cdk/aws-dynamodb';
import { IRepository } from '@aws-cdk/aws-ecr';
import { FargateService } from '@aws-cdk/aws-ecs';
import { CfnManagedPolicy, Effect, PolicyDocument, PolicyStatement, User } from '@aws-cdk/aws-iam';
import { IFunction } from '@aws-cdk/aws-lambda';
import { IBucket } from '@aws-cdk/aws-s3';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, Stack, StackProps } from '@aws-cdk/core';
import { DEV_ACCOUNT_ID } from '../consts';

interface JenkinsDeploysService {
  /** The Fargate Service to deploy. */
  service: FargateService;
  /** The ECR repository for the service's container image. */
  repo: IRepository;
  /** The secret management ARN */
  secret?: ISecret;
}

interface JenkinsDeploysLambdaFunction {
  /** The Lambda Function to deploy. */
  lambdaFunction: IFunction;
  /** The S3 Bucket for the service's container image. */
  bucket: IBucket;
}

interface JenkinsDeploymentsStackProps extends StackProps {
  /** Lambda Function and S3 bucket that needs to be accessed by jenkins */
  lambdaFunctions: JenkinsDeploysLambdaFunction[];
  /** ECS services that need to be accessed by Jenkins. */
  services: JenkinsDeploysService[];
  /** The DynamoDB allowlist table to use for E2E tests. */
  allowlist?: ITable;
}

/**
 * Creates a Jenkins user and gives it all needed permissions to manage deploys for the account.
 */
export class JenkinsDeploysStack extends Stack {
  constructor(scope: Construct, name: string, props: JenkinsDeploymentsStackProps) {
    super(scope, name, props);

    const user = new User(this, 'JenkinsUser');

    // Grant global permissions
    user.addToPolicy(
      new PolicyStatement({
        actions: [
          'ecr:DescribeRepositories',
          'ecr:GetAuthorizationToken',
          'ecs:DeregisterTaskDefinition',
          'ecs:DescribeTaskDefinition',
          'ecs:ListTaskDefinitions',
          'ecs:RegisterTaskDefinition',
          'tag:GetResources',
        ],
        resources: ['*'],
        effect: Effect.ALLOW,
      })
    );

    const secrets: ISecret[] = [];

    // Grant needed permissions for each service
    for (const { repo, service, secret } of props.services) {
      // Grant ECR permissions
      repo.grant(
        user,
        'ecr:BatchCheckLayerAvailability',
        'ecr:CompleteLayerUpload',
        'ecr:InitiateLayerUpload',
        'ecr:PutImage',
        'ecr:UploadLayerPart'
      );

      // Grant ECS permissions
      user.addToPolicy(
        new PolicyStatement({
          actions: [
            'ecs:DescribeServices',
            'ecs:DescribeTasks',
            'ecs:ListServices',
            'ecs:ListTasks',
            'ecs:UpdateService',
          ],
          resources: ['*'],
          conditions: {
            StringEquals: { 'ecs:cluster': service.cluster.clusterArn },
          },
          effect: Effect.ALLOW,
        })
      );

      // Grant ECS permissions
      user.addToPolicy(
        new PolicyStatement({
          actions: ['ecs:DescribeClusters'],
          resources: [service.cluster.clusterArn],
          effect: Effect.ALLOW,
        })
      );

      // Grant task definition role permissions
      service.taskDefinition.taskRole.grantPassRole(user);

      if (service.taskDefinition.executionRole) {
        service.taskDefinition.executionRole.grantPassRole(user);
      }

      if (!!secret) {
        // when secret shows up, allow jenkins user to access secrect value
        secrets.push(secret);
      }
    }

    // Grant needed permissions for each lambda funciton
    for (const { bucket, lambdaFunction } of props.lambdaFunctions) {
      // Grant S3 Bucket permissions
      user.addToPolicy(
        new PolicyStatement({
          actions: ['s3:Abort', 's3:PutObject'],
          resources: [bucket.bucketArn, `${bucket.bucketArn}/*`],
          effect: Effect.ALLOW,
        })
      );

      // Grant lambda function permissions
      user.addToPolicy(
        new PolicyStatement({
          actions: ['lambda:UpdateFunctionCode'],
          resources: [lambdaFunction.functionArn],
          effect: Effect.ALLOW,
        })
      );
    }

    this.addE2EPermissionToDevJenkinsUser(user, secrets, props.allowlist);
  }

  // the following method is for granting permissions to dev Jenkins to run e2e tests (the e2e tests always use aws credentials of dev account)
  private addE2EPermissionToDevJenkinsUser(user: User, secrets: ISecret[], allowlist?: ITable) {
    if (this.account !== DEV_ACCOUNT_ID) {
      return;
    }

    const policyDocument = new PolicyDocument();
    policyDocument.addStatements(
      // policy used by S2S client: https://wiki.twitch.com/pages/viewpage.action?pageId=183990900
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['sts:AssumeRole'],
        resources: ['arn:aws:iam::180116294062:role/malachai/*'],
      }),

      // always grant SNS topics access (dev account) to e2e test client topic
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['sns:Publish'],
        resources: [
          `arn:aws:sns:us-west-2:${this.account}:game-21779_client-wffpdvstzkmrmosg7je1yd9168g6as_env-dev`,
          `arn:aws:sns:us-west-2:${this.account}:DevSnsTopics-DevMetadataA232980F-9BEUNU5O3DSD`,
          `arn:aws:sns:us-west-2:${this.account}:DevSnsTopics-ProdMetadata37C49FE2-18CPAMEA4HBCY`,
        ],
      }),

      // grant access to secret management
      ...secrets.map(
        secret =>
          new PolicyStatement({
            effect: Effect.ALLOW,
            actions: ['secretsmanager:GetSecretValue'],
            resources: [secret.secretArn],
          })
      )
    );

    // grant access to DynamoDB allowlist
    if (!!allowlist) {
      policyDocument.addStatements(
        new PolicyStatement({
          effect: Effect.ALLOW,
          actions: ['dynamodb:GetItem'],
          resources: [allowlist.tableArn],
        })
      );
    }

    const managePolicy = new CfnManagedPolicy(this, 'DevJenkinsE2ETestManagedPolicy', {
      policyDocument,
    });

    user.addManagedPolicy({
      // why using managePolicy.ref?
      // reference: https://github.com/aws/aws-cdk/blob/69bff3d28687f7db1fb65ef4c967f8cce3b554ec/packages/%40aws-cdk/core/lib/resource.ts#L157-L170
      managedPolicyArn: managePolicy.ref,
    });
  }
}
