import { AccountPrincipal, Effect, ManagedPolicy, PolicyStatement, Role } from '@aws-cdk/aws-iam';
import { Topic } from '@aws-cdk/aws-sns';
import { UrlSubscription } from '@aws-cdk/aws-sns-subscriptions';
import { Construct, Stack, StackProps } from '@aws-cdk/core';
import { LEAGUE_GAME_ID, SNS_TOPIC_ENVS, SNS_TOPICS_METADATA } from '../consts';

interface SnsTopicsStackProps extends StackProps {
  /** Pager Duty url */
  pagerDutyURL?: string;
  /** Graffiti account id */
  graffitiAccountID: string;
  /** CME League account id */
  leagueCMEAccountIDs: string[];
  /** All access account id */
  allAccessAccountIDs: string[];
  /** E2-Ingest-HTTP account id */
  e2IngestHttpAccountID: string;
}

/**
 * SNS Topics for the Ingest Service.
 */
export class SnsTopicsStack extends Stack {
  public readonly alertsTopic: Topic;
  public readonly devMetadataTopic: Topic;
  public readonly devTagsTopic: Topic;
  public readonly prodMetadataTopic: Topic;
  public readonly prodTagsTopic: Topic;

  constructor(scope: Construct, name: string, props: SnsTopicsStackProps) {
    super(scope, name, props);

    const subscriptionPrinciples = props.allAccessAccountIDs.map(id => new AccountPrincipal(id));
    this.alertsTopic = new Topic(this, 'Alerts');
    if (props.pagerDutyURL) {
      this.alertsTopic.addSubscription(new UrlSubscription(props.pagerDutyURL));
    }

    this.devMetadataTopic = new Topic(this, 'DevMetadata');
    this.prodMetadataTopic = new Topic(this, 'ProdMetadata');
    this.devTagsTopic = new Topic(this, 'DevTags');
    this.prodTagsTopic = new Topic(this, 'ProdTagsV2');
    this.prodTagsTopic.addToResourcePolicy(
      new PolicyStatement({
        effect: Effect.ALLOW,
        actions: ['sns:Subscribe'],
        principals: [new AccountPrincipal(props.graffitiAccountID)],
        resources: [this.prodTagsTopic.topicArn],
        conditions: {
          StringEquals: { 'sns:Protocol': 'sqs' },
        },
      })
    );

    for (const [gameID, values] of Object.entries(SNS_TOPICS_METADATA)) {
      for (const clientId of values) {
        for (const env of SNS_TOPIC_ENVS) {
          const idAndName = `game-${gameID}_client-${clientId}_env-${env}`;
          const gameTopicDev = new Topic(this, idAndName, { topicName: idAndName });
          if (gameID === LEAGUE_GAME_ID) {
            gameTopicDev.addToResourcePolicy(
              new PolicyStatement({
                effect: Effect.ALLOW,
                actions: ['sns:Subscribe'],
                principals: props.leagueCMEAccountIDs
                  .map(id => new AccountPrincipal(id))
                  .concat(subscriptionPrinciples),
                resources: [gameTopicDev.topicArn],
                conditions: {
                  StringEquals: { 'sns:Protocol': 'sqs' },
                },
              })
            );
          } else if (subscriptionPrinciples.length > 0) {
            gameTopicDev.addToResourcePolicy(
              new PolicyStatement({
                effect: Effect.ALLOW,
                actions: ['sns:Subscribe'],
                principals: subscriptionPrinciples,
                resources: [gameTopicDev.topicArn],
                conditions: {
                  StringEquals: { 'sns:Protocol': 'sqs' },
                },
              })
            );
          }

          /** Subscription and publish rights for E2-Ingest-HTTP */
          gameTopicDev.addToResourcePolicy(
            new PolicyStatement({
              effect: Effect.ALLOW,
              actions: ['sns:Subscribe', 'sns:Publish'],
              principals: [new AccountPrincipal(props.e2IngestHttpAccountID)],
              resources: [gameTopicDev.topicArn],
            })
          );
        }
      }
    }

    // Cross-account role that allows E2 Ingest HTTP service to manage SNS topics.
    new Role(this, 'IAMRole', {
      assumedBy: new AccountPrincipal(props.e2IngestHttpAccountID),
      managedPolicies: [ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess')],
    });
  }
}
