import { Alarm } from '@aws-cdk/aws-cloudwatch';
import { Effect, PolicyStatement } from '@aws-cdk/aws-iam';
import { Code, Function as LambdaFunction, Runtime } from '@aws-cdk/aws-lambda';
import { IBucket } from '@aws-cdk/aws-s3';
import { ITopic } from '@aws-cdk/aws-sns';
import { LambdaSubscription } from '@aws-cdk/aws-sns-subscriptions';
import { Construct, Duration, Stack, StackProps, Tag } from '@aws-cdk/core';

interface TagsValidatorStackProps extends StackProps {
  /** SNS Topic for dev tags */
  tagsDevTopic: ITopic;
  /** SNS Topic for prod tags */
  tagsProdTopic: ITopic;
  /** SNS Topic for metadata */
  metadataTopic: ITopic;
  /** SNS Topic for alerts */
  alertsTopic: ITopic;
  /** Bucket containing the function */
  bucket: IBucket;
}

/**
 * Creates lambda function for the Tag Validator.
 */
export class TagsValidatorStack extends Stack {
  public readonly tagsValidatorLambda: LambdaFunction;
  public readonly errorsAlarm: Alarm;

  constructor(scope: Construct, name: string, props: TagsValidatorStackProps) {
    super(scope, name, props);

    this.tagsValidatorLambda = new LambdaFunction(this, 'TagsValidator', {
      code: Code.bucket(props.bucket, 'function.zip'), // packed and deployed from https://git-aws.internal.justin.tv/devhub/mdaas-discovery-tags-validator
      environment: {
        DEV_TAGS_TOPIC_ARN: props.tagsDevTopic.topicArn,
        PROD_TAGS_TOPIC_ARN: props.tagsProdTopic.topicArn,
      },
      handler: 'main',
      initialPolicy: [
        new PolicyStatement({
          effect: Effect.ALLOW,
          actions: ['sns:Publish'],
          resources: [props.tagsDevTopic.topicArn, props.tagsProdTopic.topicArn],
        }),
        new PolicyStatement({
          effect: Effect.ALLOW,
          actions: ['cloudwatch:PutMetricData'],
          resources: ['*'],
        }),
      ],
      runtime: Runtime.GO_1_X,
      timeout: Duration.seconds(10),
    });

    props.metadataTopic.addSubscription(new LambdaSubscription(this.tagsValidatorLambda));

    this.errorsAlarm = new Alarm(this, 'ErrorsAlarm', {
      alarmDescription: 'Notify pagerduty if lambda has consecutive errors for 3 periods.',
      evaluationPeriods: 3,
      metric: this.tagsValidatorLambda.metricErrors(),
      threshold: 1,
    });

    this.errorsAlarm.addAlarmAction({
      bind: () => ({ alarmActionArn: props.alertsTopic.topicArn }),
    });

    Tag.add(this.tagsValidatorLambda, 'DeployLambda', 'TagsValidator');
  }
}
