package e2e_utils

import (
	"encoding/json"
	"fmt"
	"os"

	"github.com/davecgh/go-spew/spew"

	"github.com/fatih/color"

	. "code.justin.tv/devhub/mdaas-ingest/models"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	awskinesis "github.com/aws/aws-sdk-go/service/kinesis"
)

type kinesisResultType string

const (
	kinesisFirstFullState   kinesisResultType = "kinesis_first_full_state"
	kinesisGeneralFullState kinesisResultType = "kinesis_general_full_state"
	kinesisDeltaState       kinesisResultType = "kinesis_delta_state"
)

type kinesisResult struct {
	Type          kinesisResultType
	FirstKeyFrame *string
	LastFullState *string
}

type kinesisRecord struct {
	event Event
	seq   *string
}

func (tester *eceImpl) openKinesisSession(broadcasterID string, gameID string) *awskinesis.Kinesis {
	sess, err := session.NewSession()
	if err != nil {
		fmt.Println("Error connecting to kinesis ", err)
		os.Exit(1)
	}

	conf := &aws.Config{Region: aws.String("us-west-2")}
	return awskinesis.New(sess, conf)
}

func (tester *eceImpl) getIterator(broadcasterID, gameID string) (*awskinesis.Kinesis, string) {
	k := tester.openKinesisSession(broadcasterID, gameID)
	shardName, shardID := getShardNameAndID(gameID)

	output, err := k.GetShardIterator(&awskinesis.GetShardIteratorInput{
		ShardId:           aws.String(shardID),
		ShardIteratorType: aws.String(awskinesis.ShardIteratorTypeLatest),
		StreamName:        aws.String(shardName),
	})

	if err != nil {
		fmt.Println("Error setting up kinesis iterator: ", err.Error())
	}

	if output == nil || output.ShardIterator == nil {
		color.Red("Error creating iterator")
		os.Exit(1)
	}

	return k, *output.ShardIterator
}

func getEvents(k *awskinesis.Kinesis, iterator *string) []kinesisRecord {
	var actualEvents []kinesisRecord
	records, err := k.GetRecords(&awskinesis.GetRecordsInput{
		ShardIterator: iterator,
		Limit:         aws.Int64(int64(100)),
	})

	if err != nil {
		fmt.Println("Error pulling kinesis records: ", err.Error())
	}

	if records.Records != nil && len(records.Records) > 0 {
		for _, r := range records.Records {
			event := Event{}
			err := json.Unmarshal(r.Data, &event)
			if err != nil {
				fmt.Println("Error parsing kinesis records: ", err.Error())
			}

			record := kinesisRecord{event: event}
			record.seq = r.SequenceNumber

			actualEvents = append(actualEvents, record)
		}
	}
	return actualEvents
}

func validateKinesisDeltaStateEvent(e Event, kinesisResult kinesisResult, expected expectedKinesisProps) {
	if e.Full != nil || e.Delta == nil || e.Delta.SeqStart == "" || e.Delta.BroadcasterID != expected.BroadcasterId || e.Delta.GameID != expected.GameID || e.Delta.Data == nil {
		fmt.Println("Error validating delta kinesis event")
		spew.Dump(e)
		os.Exit(1)
	}
	if kinesisResult.FirstKeyFrame == nil || e.Delta.FirstKeyFrame != *kinesisResult.FirstKeyFrame {
		fmt.Println("Error validating delta kinesis event")
		spew.Dump(e)
		os.Exit(1)
	}
	if kinesisResult.LastFullState == nil || e.Delta.SeqStart != *kinesisResult.LastFullState {
		fmt.Println("Error validating delta kinesis event")
		spew.Dump(e)
		os.Exit(1)
	}
	color.Yellow("kinesis delta state validated")
}

type expectedKinesisProps struct {
	IsServerData  bool
	GameID        string
	BroadcasterId string
}

func validateKinesisFullStateEvent(e Event, kinesisResult kinesisResult, expected expectedKinesisProps) {
	if e.Delta != nil || e.Full == nil || e.Full.BroadcasterID != expected.BroadcasterId || e.Full.GameID != expected.GameID || e.Full.Data == nil || e.Full.ClientID == "" || e.Full.Env != "dev" || e.Full.IsServerData != expected.IsServerData {
		color.Red("Error validating full state kinesis event")
		spew.Dump(e)
		os.Exit(1)
	}
	if kinesisResult.Type == kinesisFirstFullState {
		if kinesisResult.FirstKeyFrame != nil {
			color.Red("Error validating first full state kinesis event")
			spew.Dump(e)
			os.Exit(1)
		}
	} else {
		if kinesisResult.FirstKeyFrame == nil || e.Full.FirstKeyFrame != *kinesisResult.FirstKeyFrame {
			color.Red("Error validating general full state kinesis event")
			spew.Dump(e)
			os.Exit(1)
		}
	}
	color.Yellow("kinesis full state validated")
}

func validateSequence(expectedEvents []kinesisResult, actualEvents []kinesisRecord, expected expectedKinesisProps) {
	var firstKeyFrame *string
	var lastFullState *string
	for i, event := range expectedEvents {
		if i >= len(actualEvents) {
			color.Red("Actual events less than expected")
			os.Exit(1)
		}
		event.FirstKeyFrame = firstKeyFrame
		event.LastFullState = lastFullState
		switch event.Type {
		case kinesisDeltaState:
			validateKinesisDeltaStateEvent(actualEvents[i].event, event, expected)
		case kinesisGeneralFullState:
			validateKinesisFullStateEvent(actualEvents[i].event, event, expected)
			lastFullState = actualEvents[i].seq
		case kinesisFirstFullState:
			event.FirstKeyFrame = nil
			event.LastFullState = nil
			validateKinesisFullStateEvent(actualEvents[i].event, event, expected)
			firstKeyFrame = actualEvents[i].seq
			lastFullState = actualEvents[i].seq
		default:
			color.Red("Receiving wrong expected kinesis event type")
			os.Exit(1)
		}
	}
}

// TODO: work with reading from controller, now it is only hard coded
func getShardNameAndID(gameID string) (string, string) {
	return "game.21779.1", "shardId-000000000000"
}

func buildStandardExpectedKinesisResults(num int) []kinesisResult {
	var resultArray = make([]kinesisResult, num)
	for i := range resultArray {
		if i == 0 {
			resultArray[i] = kinesisResult{
				Type: kinesisFirstFullState,
			}
		} else if i%10 == 0 {
			resultArray[i] = kinesisResult{
				Type: kinesisGeneralFullState,
			}
		} else {
			resultArray[i] = kinesisResult{
				Type: kinesisDeltaState,
			}
		}
	}
	return resultArray
}
