package e2e_utils

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"time"

	jdelta "code.justin.tv/devhub/json-delta/lib"

	. "code.justin.tv/devhub/twitch-e2-ingest/models"
	"golang.org/x/net/websocket"
)

func (tester *eceImpl) TestSNSPublishNormalPath() {
	connectPack := replaceToken(tester.UserToken, StandardUserConnectionData)

	connectTime := time.Now().UnixNano()
	// connect
	ws, err := tester.connect()
	if err != nil {
		log.Fatal(err)
	}

	defer closeConnection()("TestSNSPublishNormalPath", ws)

	spinUpValidationWorker(validateSNSHappyPath, ws)

	var connectData map[string]interface{}
	_ = json.Unmarshal([]byte(connectPack), &connectData)

	// Send connect pack
	err = websocket.JSON.Send(ws, connectData)
	if err != nil {
		fmt.Println("Error sending connect message: ", err.Error())
	}

	var deltaData1 map[string]interface{}
	_ = json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), &deltaData1)

	for i := 0; i < 3; i++ {
		// Send delta pack
		err = websocket.JSON.Send(ws, deltaData1)
		if err != nil {
			fmt.Println("Error sending delta ops message: ", err.Error())
		}
	}

	for i := 0; i < 2; i++ {
		// Send bad delta pack
		var deltaData1 map[string]interface{}
		_ = json.Unmarshal([]byte(BadDeltaPack), &deltaData1)
		err = websocket.JSON.Send(ws, deltaData1)
		if err != nil {
			fmt.Println("Error sending delta ops message: ", err.Error())
		}
	}

	// Send refresh package
	var refreshData map[string]interface{}
	_ = json.Unmarshal([]byte(StandardRefresh), &refreshData)

	// Send refresh pack
	err = websocket.JSON.Send(ws, refreshData)
	if err != nil {
		fmt.Println("Error sending refresh message: ", err.Error())
	}

	// Wait 3 seconds for possible SQS delays
	time.Sleep(3 * time.Second)

	var actualEvents = receiveSQSMessages(stagingSQS)
	resp, _ := getClientInfo(tester.Token)

	cData, _ := (connectData["connect"]).(map[string]interface{})
	fullData, _ := cData["data"].(map[string]interface{})
	originalData := deepCopy(fullData)
	delta, _ := (deltaData1["delta"]).([]interface{})
	updated, _ := jdelta.PatchDeltas(fullData, delta)

	var refresh map[string]interface{}
	_ = json.Unmarshal([]byte(StandardRefresh), &refresh)
	rData, _ := (refresh["refresh"]).(map[string]interface{})

	// Compose result validations
	initialData := GameFullData{
		ClientID:       resp.ClientID,
		Env:            "dev",
		BroadcasterIDs: []string{tester.BroadcasterIDs},
		GameID:         tester.GameID,
		Time:           time.Now().UnixNano(),
		ConnectionID:   connectTime,
		MessageID:      0,
		SessionID:      sessionID,
		Data:           originalData,
	}

	deltaData := GameFullData{
		ClientID:       resp.ClientID,
		Env:            "dev",
		BroadcasterIDs: []string{tester.BroadcasterIDs},
		GameID:         tester.GameID,
		Time:           time.Now().UnixNano(),
		ConnectionID:   connectTime,
		MessageID:      0,
		SessionID:      sessionID,
		Data:           updated,
	}

	refreshFullData := GameFullData{
		ClientID:       resp.ClientID,
		Env:            "dev",
		BroadcasterIDs: []string{tester.BroadcasterIDs},
		GameID:         tester.GameID,
		Time:           time.Now().UnixNano(),
		ConnectionID:   connectTime,
		MessageID:      0,
		SessionID:      sessionID,
		Data:           rData["data"],
	}

	expectedEvents := []GameFullData{
		initialData,
		deltaData,
		deltaData,
		deltaData,
		refreshFullData,
	}

	validateSQSSequence(expectedEvents, actualEvents)
}

func getClientInfo(token string) (AuthResp, error) {
	httpClient := &http.Client{}

	var authResult AuthResp
	// Currently it is only talking to id.twitch.tv, make it variable if we find needs
	req, err := http.NewRequest("GET", "https://id.twitch.tv/oauth2/validate", nil)
	if err != nil {
		return authResult, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))

	resp, err := httpClient.Do(req)
	if err != nil {
		return authResult, err
	} else if resp == nil {
		return authResult, errors.New("cant load client info")
	}

	if resp.StatusCode >= 400 {
		return authResult, errors.New("cant load client info")
	}

	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return authResult, err
	}

	err = json.Unmarshal(body, &authResult)
	return authResult, err
}

func validateSNSHappyPath(count int, result map[string]interface{}) {
	fmt.Println("Received ", result)
	switch count {
	case 1:
		validateConnectPack(result)
	case 2:
		validateResp(result, DeltaInvalid, "delta invalid message")
	case 3:
		validateResp(result, DeltaInvalid, "delta invalid message")
	default:
		encounterUnexpectedResp(result)
	}
}
