package main

import (
	"crypto/tls"
	"flag"
	"fmt"
	"os"
	"strconv"
	"time"

	"github.com/rollbar/rollbar-go"
	log "github.com/sirupsen/logrus"

	metricsmiddleware "code.justin.tv/amzn/TwitchTelemetryMetricsMiddleware"
	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/mdaas-ingest/config"
	"code.justin.tv/devhub/mdaas-ingest/internal/clients/kinesis"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"
	ingestlogger "code.justin.tv/devhub/mdaas-ingest/logger"
	"code.justin.tv/devhub/mdaas-ingest/server"
	"code.justin.tv/devhub/twitch-e2-ingest/authentication"
	"code.justin.tv/devhub/twitch-e2-ingest/dynamo"
	"code.justin.tv/devhub/twitch-e2-ingest/sns"
	"code.justin.tv/karlpatr/message-prototype/libs/logging"
	"code.justin.tv/karlpatr/message-prototype/libs/timeout"
	"code.justin.tv/karlpatr/message-prototype/libs/websocket"
	"code.justin.tv/security/ephemeralcert"
	"code.justin.tv/video/metrics-middleware/v2/operation"
)

const (
	defaultHealthCheck = "/ping"
	defaultPort        = 80
)

func checkEnv(key, def string) string {
	if value, ok := os.LookupEnv(key); ok {
		return value
	}
	return def
}

func checkPort() int64 {
	if value, ok := os.LookupEnv("PORT"); ok {
		port, err := strconv.ParseInt(value, 10, 16)
		if err != nil {
			panic("Illegal port: " + value)
		}
		return port
	}

	return defaultPort
}

func main() {
	port := flag.Int64("port", checkPort(), "port for incoming connections")
	healthCheck := flag.String("health-check", checkEnv("HEALTH_CHECK", defaultHealthCheck), "health check path")
	logLevel := flag.String("log", "info", "set logging level: [trace,debug,info,warning,error]")
	enableCert := flag.Bool("cert", false, "if self-cert enabled in the server")
	fanout := flag.Bool("fanout-kinesis", false, "if fan out to kinesis")

	// Value: production, staging, development
	env := os.Getenv("ENV")

	// If canary
	canary := ""
	if os.Getenv("CANARY") == "true" {
		canary = "-canary"
	}

	flag.Parse()

	mgr := lifecycle.NewManager()

	err := config.LoadConfigs(env, fanout)
	if err != nil {
		log.Fatal("Exiting on error: ", err, " when loading configs.")
	}

	conf := config.GetConfigs()

	// Set up rollbar
	setupRollbar(conf, canary)

	setUpLogger(*logLevel)
	logger := ingestlogger.Log

	defer exit(mgr, logger)

	var tlsConfig *tls.Config
	if enableCert != nil && *enableCert {
		tlsConfig, err = ephemeralcert.GenerateTLSConfig()
		if err != nil {
			log.Fatal(logging.Error, "Unable to load TLS", err)
		}
	}

	settings := &websocket.Settings{
		Certs:       tlsConfig,
		Lifecycle:   mgr,
		Logger:      logger,
		Timeout:     timeout.NewConstantSampler(195 * time.Second),
		OnTimeout:   websocket.Disconnect,
		HealthCheck: healthCheck,
	}

	// Start on metrics reporting
	fLogger := &metrics.FLogger{}

	if err := metrics.InitializeProcessIdentifier(fmt.Sprintf("ingest%s", canary), conf.Env); err != nil {
		log.Fatal("error initializing process identifier", err.Error())
	}
	processorIdentifier := metrics.ProcessIdentifier()
	metrics.InitializeMetrics(fLogger, &processorIdentifier)

	// Disable until we find usage
	// metrics.StartGoStatsCollection(fLogger)
	metrics.StartSystemStatsCollection(fLogger, &processorIdentifier, metrics.Observer())

	// initialize kinesis client
	kc, err := kinesis.NewPublisher(&kinesis.Config{DebugLogsFirehose: conf.DebugLogsFirehose})
	if err != nil {
		log.Fatal("Kinesis Client init failed with error: ", err)
		return
	}

	snsc, err := sns.NewPublisher(&sns.Config{
		DefaultTopicARN: conf.DevSNSArn,
	})
	if err != nil {
		log.Fatal("SNS Client init failed with error: ", err)
		return
	}

	// initialize DynamoDB allowlist
	opMonitor := &metricsmiddleware.OperationMonitor{
		SampleReporter: metrics.Reporter().SampleReporter,
		AutoFlush:      false,
	}
	allowlist, err := dynamo.NewAllowlist(&dynamo.Config{
		AWSRegion: "us-west-2",
		TableName: conf.AllowlistTableName,
		OperationStarter: &operation.Starter{
			OpMonitors: []operation.OpMonitor{opMonitor},
		},
	}, snsc)
	if err != nil {
		log.Fatal("Game Client Allowlist init failed with error: ", err)
		return
	}

	// initialize auth session
	authClient, err := authentication.NewClient(
		conf.OwlHost,
		conf.InterpolHost,
		conf.S2SCallerName,
		allowlist,
		ingestlogger.StandardLogger(),
	)
	if err != nil {
		log.Fatal("Exiting on error: ", err, " when loading configs.")
	}

	serverPort := int(*port)
	// Override port if local
	if env == "development" {
		serverPort = 8080
	}

	bindings := server.Bindings{Bindings: make(map[string]*server.Binding)}
	service, err := websocket.NewServiceFactory(serverPort, settings)(server.NewServer(kc, snsc, authClient, &bindings).Factory())
	if err != nil {
		logger(logging.Error, "Exiting on error:", err)
		return
	}

	service.Start()

	// Listen for interrupt signal
	logger(logging.Info, "Exiting on signal:", mgr.ListenForInterrupt())
	service.Stop()

	// Shutdown behavior, send reconnect messages
	bindings.Lock()

	for _, val := range bindings.Bindings {
		// send reconnect message to all live connections in parallel
		go val.WriteReconnectUponServiceClosure()
	}

	bindings.Unlock()
	logger(logging.Info, "Waiting for connections to drop (max 10 seconds)")
	service.WaitForDrainingConnections(time.Now().Add(10 * time.Second))
}

func exit(mgr lifecycle.Manager, logger logging.Function) {
	if err := mgr.ExecuteAll(); err != nil {
		logger(logging.Error, "Shutdown error:", err)
	}
}

func setUpLogger(logLevel string) {
	if logLevel == "debug" || logLevel == "trace" {
		log.SetLevel(log.DebugLevel)
	}
}

func setupRollbar(cfg config.Config, canary string) {
	ingestlogger.Configure(cfg, canary)
	rollbar.Wait()
}
