package config

import (
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

// for validating config loading results
const (
	devSecret              = "PrimarySecret2E899820-wBmIN3IbSbot"
	prodSecret             = "PrimarySecret2E899820-PrqQvK35tmS4"
	devMetadataDevSNSArn   = "arn:aws:sns:us-west-2:797743463538:DevSnsTopics-DevMetadataA232980F-9BEUNU5O3DSD"
	devMetadataProdSNSArn  = "arn:aws:sns:us-west-2:797743463538:DevSnsTopics-ProdMetadata37C49FE2-18CPAMEA4HBCY"
	prodMetadataDevSNSArn  = "arn:aws:sns:us-west-2:433343386846:ProdSnsTopics-DevMetadataA232980F-VMHOJETYGBO6"
	prodMetadataProdSNSArn = "arn:aws:sns:us-west-2:433343386846:ProdSnsTopics-ProdMetadata37C49FE2-1WZHL3SY7KV7O"
	// local could not use a2z address since it is not accessible locally
	owlDevelopmentHost = "https://owl.prod.internal.us-west-2.twitch.tv"
	// TODO: change to use a2z one when i confirm with identity it is fully working
	owlStagingHost          = "https://owl.staging.internal.us-west-2.twitch.tv"
	owlProductionHost       = "https://owl.prod.internal.us-west-2.twitch.tv"
	devDebugLogsFirehose    = "DevIngestDebugLogs-debugLogsFirehose-GEZZFP7DHCKD"
	prodDebugLogsFirehose   = "ProdIngestDebugLogs-debugLogsFirehose-1WJ71QDKDCL02"
	interpolDevelopmentHost = "https://id.twitch.tv"
	interpolStagingHost     = "https://id.twitch.tv"
	interpolProductionHost  = "https://id.twitch.tv"
	devS2SCallerName        = "mdaas-ingest-dev"
	prodS2SCallerName       = "mdaas-ingest-prod"
	devAllowlistTableName   = "DevDynamoAllowlist-DynamoDBTable59784FC0-5CKDVKPZMHLP"
	prodAllowlistTableName  = "ProdDynamoAllowlist-DynamoDBTable59784FC0-10PHLZKFUC5V8"
)

type ConfigTest struct {
	suite.Suite
}

func (suite *ConfigTest) SetupTest() {
	config = Config{
		Env:           "development",
		FanoutKinesis: true,
		Secrets:       Secrets{},
	}
}

func (suite *ConfigTest) TestLoadDevConfigs() {
	err := loadEnvConfigs(dev, aws.Bool(true))
	expectedConfig := Config{
		SecretName:         devSecret,
		Secrets:            Secrets{},
		Env:                dev,
		FanoutKinesis:      true,
		DevSNSArn:          devMetadataDevSNSArn,
		ProdSNSArn:         devMetadataProdSNSArn,
		OwlHost:            owlDevelopmentHost,
		DebugLogsFirehose:  devDebugLogsFirehose,
		InterpolHost:       interpolDevelopmentHost,
		S2SCallerName:      devS2SCallerName,
		AllowlistTableName: devAllowlistTableName,
	}
	assert.Equal(suite.T(), expectedConfig, config)
	assert.NoError(suite.T(), err)
}

func (suite *ConfigTest) TestLoadStagingConfigs() {
	err := loadEnvConfigs(staging, aws.Bool(true))
	expectedConfig := Config{
		SecretName:         devSecret,
		Secrets:            Secrets{},
		Env:                staging,
		FanoutKinesis:      true,
		DevSNSArn:          devMetadataDevSNSArn,
		ProdSNSArn:         devMetadataProdSNSArn,
		OwlHost:            owlStagingHost,
		DebugLogsFirehose:  devDebugLogsFirehose,
		InterpolHost:       interpolStagingHost,
		S2SCallerName:      devS2SCallerName,
		AllowlistTableName: devAllowlistTableName,
	}
	assert.Equal(suite.T(), expectedConfig, config)
	assert.NoError(suite.T(), err)
}

func (suite *ConfigTest) TestLoadProdConfigs() {
	err := loadEnvConfigs(prod, aws.Bool(true))
	expectedConfig := Config{
		SecretName:         prodSecret,
		Secrets:            Secrets{},
		Env:                prod,
		FanoutKinesis:      true,
		DevSNSArn:          prodMetadataDevSNSArn,
		ProdSNSArn:         prodMetadataProdSNSArn,
		OwlHost:            owlProductionHost,
		DebugLogsFirehose:  prodDebugLogsFirehose,
		InterpolHost:       interpolProductionHost,
		S2SCallerName:      prodS2SCallerName,
		AllowlistTableName: prodAllowlistTableName,
	}
	assert.Equal(suite.T(), expectedConfig, config)
	assert.NoError(suite.T(), err)
}

func TestRunProcessorSuite(t *testing.T) {
	suite.Run(t, new(ConfigTest))
}
