package config

import (
	"errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/defaults"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"gopkg.in/yaml.v2"
)

const (
	prod    = "production"
	dev     = "development"
	staging = "staging"
)

func getSecret(secretName string) (string, error) {
	s, err := session.NewSession(&aws.Config{Region: aws.String("us-west-2")})
	if err != nil {
		return "", err
	}

	creds := defaults.CredChain(defaults.Config(), defaults.Handlers())

	//Create a Secrets Manager client
	svc := secretsmanager.New(s, aws.NewConfig().WithCredentials(creds).WithRegion("us-west-2"))

	input := &secretsmanager.GetSecretValueInput{
		SecretId:     aws.String(secretName),
		VersionStage: aws.String("AWSCURRENT"), // VersionStage defaults to AWSCURRENT if unspecified
	}

	result, err := svc.GetSecretValue(input)
	if err != nil {
		return "", err
	}

	var secretString string
	if result.SecretString != nil {
		secretString = *result.SecretString
	} else {
		return "", errors.New("missing secret configs")
	}

	return secretString, nil
}

func loadEnvConfigs(env string, fanout *bool) error {
	var yamlBytes []byte
	var err error
	switch env {
	case staging:
		yamlBytes, err = configDataConfigStagingYamlBytes()
		if err != nil {
			return err
		}
	case prod:
		yamlBytes, err = configDataConfigProdYamlBytes()
		if err != nil {
			return err
		}
	// for "development" and bad fallback cases
	default:
		yamlBytes, err = configDataConfigDevYamlBytes()
		if err != nil {
			return err
		}
	}

	err = yaml.Unmarshal(yamlBytes, &config)
	if err != nil {
		return err
	}

	config.Env = env

	if fanout != nil {
		config.FanoutKinesis = *fanout
	}

	return nil
}
