package kinesis

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/service/firehose"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"
	. "code.justin.tv/devhub/mdaas-ingest/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/errgroup"
)

func (p *publisher) Publish(ctx context.Context, event Event, cloneToBroadcasterInfo [][]BroadcasterKinesisInfo) ([][]BroadcasterKinesisInfo, error) {
	g, _ := errgroup.WithContext(ctx)
	updatedCloneToBroadcasterInfo := cloneToBroadcasterInfo

	for i, broadcasterInfos := range cloneToBroadcasterInfo {
		// https://golang.org/doc/faq#closures_and_goroutines
		cloneID := i + 1
		broadcasterInfos := broadcasterInfos
		// Array index indicating clone id, clone id starts from 1, and index starts from 0
		g.Go(func() error {
			streamName := event.StreamName(cloneID)
			eventsToProcess := make([]Event, len(broadcasterInfos))
			for j, info := range broadcasterInfos {
				updatedEvent := event
				updatedEvent.SetKinesisSpecificInfo(info.BroadcasterID, info.LastFullState, info.FirstKeyFrame)
				eventsToProcess[j] = updatedEvent
			}
			result := make(map[string]string)

			input, err := p.buildKinesisInput(eventsToProcess, streamName)
			if err != nil {
				return errors.New("fail to build input for kinesis")
			}

			for i := 0; i < p.Conf.retryCount(); i++ {
				output, err := p.Client.PutRecords(&input)
				// Log kinesis errors
				if err != nil {
					log.Error(err)
				}

				if output == nil || len(output.Records) == 0 {
					// Go to next retry directly
					continue
				}

				// When output is not empty, retry the left ones
				var eventsToRetry []Event
				for j, e := range eventsToProcess {
					if j < len(output.Records) {
						record := output.Records[j]
						if record.SequenceNumber != nil {
							result[e.BroadcasterID()] = *record.SequenceNumber
							// skip to add to entry to retry if actually succeed
							continue
						} else if record.ErrorCode != nil {
							// Report errors
							metrics.Reporter().Report(fmt.Sprintf("kinesis_error_%s", *record.ErrorCode), 1.0, telemetry.UnitCount)
						}
					}

					eventsToRetry = append(eventsToRetry, eventsToProcess[j])
				}

				eventsToProcess = eventsToRetry

				// We're done early if there are no errors and no events to process
				if len(eventsToProcess) == 0 {
					break
				}

				time.Sleep(p.Conf.retryDelay())
			}

			if len(eventsToProcess) != 0 {
				return errors.New("unable to publish data to kinesis")
			}

			// if this is a full state, update broadcaster kinesis info
			if event.Full != nil {
				// https://godoc.org/golang.org/x/sync/errgroup
				infoArray := updatedCloneToBroadcasterInfo[cloneID-1]
				updatedKinesisInfo := make([]BroadcasterKinesisInfo, len(infoArray))
				for i, broadcasterInfos := range infoArray {
					updatedInfo := broadcasterInfos
					updatedInfo.LastFullState = result[updatedInfo.BroadcasterID]
					// if first key frame is empty, indicating this is the first publish, replace this field
					if updatedInfo.FirstKeyFrame == "" {
						updatedInfo.FirstKeyFrame = result[updatedInfo.BroadcasterID]
					}
					updatedKinesisInfo[i] = updatedInfo
				}
				updatedCloneToBroadcasterInfo[cloneID-1] = updatedKinesisInfo
			}

			return nil
		})
	}

	// if there is an error, return original kinesis info data and the error
	if err := g.Wait(); err != nil {
		return cloneToBroadcasterInfo, err
	}

	return updatedCloneToBroadcasterInfo, nil
}

func (p *publisher) buildKinesisInput(eventsToProcess []Event, streamName string) (kinesis.PutRecordsInput, error) {
	var records []*kinesis.PutRecordsRequestEntry
	for _, e := range eventsToProcess {
		jsonBlob, err := json.Marshal(e)
		if err != nil {
			return kinesis.PutRecordsInput{}, err
		}

		record := &kinesis.PutRecordsRequestEntry{
			Data:         jsonBlob,
			PartitionKey: aws.String(e.PartitionKey()),
		}
		records = append(records, record)
	}

	input := kinesis.PutRecordsInput{
		Records:    records,
		StreamName: aws.String(streamName),
	}

	return input, nil
}

// SavePublishedStateDataInS3 is to put log into firehose
// if publishing fails again, swallow the error in order not to affect the main function
// log the error message to cloud watch
func (p *publisher) SavePublishedStateDataInS3(message string) {
	record := &firehose.Record{
		Data: []byte(message),
	}
	input := &firehose.PutRecordInput{
		DeliveryStreamName: &p.Conf.DebugLogsFirehose,
		Record:             record,
	}
	_, putRecordErr := p.Firehose.PutRecord(input)
	if putRecordErr != nil {
		log.Errorf("Error when calling Firehose.PutRecord: %v\n", putRecordErr)
	}
}
