package kinesis

import (
	"context"

	"github.com/pkg/errors"

	. "code.justin.tv/devhub/mdaas-ingest/test_utils"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

// Assuming 2 clones for all the tests below
func (suite *PublisherTest) TestFullStatePublishInTheMiddle() {
	mapping := GetCloneToBroadcasterMapping(NormalMessage)
	updatedMapping := GetCloneToBroadcasterMapping(UpdatedMessage)
	output1 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[0][0].LastFullState)},
			{SequenceNumber: aws.String(updatedMapping[0][1].LastFullState)},
		},
	}

	output2 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[1][0].LastFullState)},
			{SequenceNumber: aws.String(updatedMapping[1][1].LastFullState)},
		},
	}

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.1"),
		Records:    GetKinesisInputs(false, false, 1, mapping),
	}).Return(output1, nil)

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.2"),
		Records:    GetKinesisInputs(false, false, 2, mapping),
	}).Return(output2, nil)

	updatedInfo, err := suite.publisher.Publish(context.Background(), GetEventFromIngest(false), mapping)

	suite.NoError(err)
	suite.Equal(updatedMapping, updatedInfo)
}

func (suite *PublisherTest) TestFullStatePublishInAtBeginning() {
	mapping := GetCloneToBroadcasterMapping(FirstMessage)
	updatedMapping := GetCloneToBroadcasterMapping(FirstKeyEqualLastFullState)
	output1 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[0][0].FirstKeyFrame)},
			{SequenceNumber: aws.String(updatedMapping[0][1].FirstKeyFrame)},
		},
	}

	output2 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[1][0].FirstKeyFrame)},
			{SequenceNumber: aws.String(updatedMapping[1][1].FirstKeyFrame)},
		},
	}

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.1"),
		Records:    GetKinesisInputs(true, false, 1, mapping),
	}).Return(output1, nil)

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.2"),
		Records:    GetKinesisInputs(true, false, 2, mapping),
	}).Return(output2, nil)

	updatedInfo, err := suite.publisher.Publish(context.Background(), GetEventFromIngest(false), mapping)

	suite.NoError(err)
	suite.Equal(updatedMapping, updatedInfo)
}

func (suite *PublisherTest) TestDeltaStatePublishInTheMiddle() {
	mapping := GetCloneToBroadcasterMapping(NormalMessage)
	updatedMapping := GetCloneToBroadcasterMapping(UpdatedMessage)
	output1 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[0][0].FirstKeyFrame)},
			{SequenceNumber: aws.String(updatedMapping[0][1].FirstKeyFrame)},
		},
	}

	output2 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[1][0].FirstKeyFrame)},
			{SequenceNumber: aws.String(updatedMapping[1][1].FirstKeyFrame)},
		},
	}

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.1"),
		Records:    GetKinesisInputs(false, true, 1, mapping),
	}).Return(output1, nil)

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.2"),
		Records:    GetKinesisInputs(false, true, 2, mapping),
	}).Return(output2, nil)

	updatedInfo, err := suite.publisher.Publish(context.Background(), GetEventFromIngest(true), mapping)

	suite.NoError(err)
	suite.Equal(mapping, updatedInfo)
}

func (suite *PublisherTest) TestDeltaStatePublishInTheMiddleWithError() {
	mapping := GetCloneToBroadcasterMapping(NormalMessage)
	updatedMapping := GetCloneToBroadcasterMapping(UpdatedMessage)
	output1 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: nil, ErrorCode: aws.String("error code")},
			{SequenceNumber: aws.String(updatedMapping[0][1].FirstKeyFrame)},
		},
	}

	output2 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[1][0].FirstKeyFrame)},
			{SequenceNumber: aws.String(updatedMapping[1][1].FirstKeyFrame)},
		},
	}

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.1"),
		Records:    GetKinesisInputs(false, true, 1, mapping),
	}).Return(output1, errors.New("unexpected error"))

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.2"),
		Records:    GetKinesisInputs(false, true, 2, mapping),
	}).Return(output2, nil)

	updatedInfo, err := suite.publisher.Publish(context.Background(), GetEventFromIngest(true), mapping)

	suite.NotNil(err)
	suite.Equal(mapping, updatedInfo)
}

func (suite *PublisherTest) TestFullStatePublishInTheMiddleWithError() {
	mapping := GetCloneToBroadcasterMapping(NormalMessage)
	updatedMapping := GetCloneToBroadcasterMapping(UpdatedMessage)
	output1 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: aws.String(updatedMapping[0][0].LastFullState)},
			{SequenceNumber: aws.String(updatedMapping[0][1].LastFullState)},
		},
	}

	output2 := &kinesis.PutRecordsOutput{
		FailedRecordCount: aws.Int64(0),
		Records: []*kinesis.PutRecordsResultEntry{
			{SequenceNumber: nil, ErrorCode: aws.String("error code")},
			{SequenceNumber: aws.String(updatedMapping[1][1].LastFullState)},
		},
	}

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.1"),
		Records:    GetKinesisInputs(false, false, 1, mapping),
	}).Return(output1, nil)

	suite.mockKinesisAPI.On("PutRecords", &kinesis.PutRecordsInput{
		StreamName: aws.String("game.testGameID1.2"),
		Records:    GetKinesisInputs(false, false, 2, mapping),
	}).Return(output2, errors.New("unexpected error"))

	updatedInfo, err := suite.publisher.Publish(context.Background(), GetEventFromIngest(false), mapping)

	suite.Error(err)
	suite.Equal(mapping, updatedInfo)
}
