package kinesis

import (
	"context"
	"time"

	"code.justin.tv/devhub/mdaas-ingest/internal/awsmocks"

	. "code.justin.tv/devhub/mdaas-ingest/models"

	settings "code.justin.tv/devhub/twitch-e2-ingest/http-settings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/firehose"
	"github.com/aws/aws-sdk-go/service/firehose/firehoseiface"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/aws/aws-sdk-go/service/kinesis/kinesisiface"
)

var cloneToGameMapping map[string]int

//go:generate go run github.com/vektra/mockery/cmd/mockery -name Publisher
type Publisher interface {
	Publish(ctx context.Context, event Event, cloneToBroadcasterInfo [][]BroadcasterKinesisInfo) ([][]BroadcasterKinesisInfo, error)
	GetCloneNumByGame(gameID string) int
	SavePublishedStateDataInS3(message string)
}

type publisher struct {
	Client   kinesisiface.KinesisAPI
	Firehose firehoseiface.FirehoseAPI
	Conf     *Config
}

func NewPublisher(c *Config) (Publisher, error) {
	// Load game to clone mapping
	var err error
	cloneToGameMapping, err = loadClones()
	if err != nil {
		return nil, err
	}

	httpSettings, err := settings.NewHTTPClientWithSettings(settings.ClientSettings{
		ConnectTimeout:        5 * time.Second,
		ExpectContinueTimeout: 1 * time.Second,
		IdleConnTimeout:       90 * time.Second,
		KeepAlive:             30 * time.Second,
		MaxIdleConns:          0,
		MaxIdleConnsPerHost:   100000,
		ResponseHeaderTimeout: 10 * time.Second,
		TLSHandshakeTimeout:   5 * time.Second,
	})

	if err != nil {
		return nil, err
	}

	sess, err := session.NewSession(&aws.Config{
		Region:     aws.String(c.awsRegion()),
		HTTPClient: httpSettings,
	})

	if err != nil {
		return nil, err
	}

	k := kinesis.New(sess)
	fh := firehose.New(sess)
	return &publisher{Client: k, Conf: c, Firehose: fh}, nil
}

func NewTestPublisher(kinesismock *awsmocks.KinesisAPI, firehoseMock *awsmocks.FirehoseAPI, c *Config) (Publisher, error) {
	// Load game to clone mapping
	var err error
	cloneToGameMapping, err = loadClones()
	if err != nil {
		return nil, err
	}
	return &publisher{Client: kinesismock, Firehose: firehoseMock, Conf: c}, nil
}
