package metrics

import (
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	log "github.com/sirupsen/logrus"
)

// Reporter reports metrics.
// It differs from telemetry.SampleReporter in that it lets you specify per metric dimensions.
type ReporterWithDimensions struct {
	SampleReporter telemetry.SampleReporter
}

type DimensionSet map[string]string

func (r *ReporterWithDimensions) Report(name string, value float64, units string, dimensions ...DimensionSet) {
	if len(dimensions) == 0 {
		r.SampleReporter.Report(name, value, units)
		return
	}

	builder := r.builderWithDimensions(dimensions)
	// create sample
	sample, err := builder.Build(name, value, units)
	if err != nil {
		log.Error(err)
		return
	}

	// observe sample
	r.SampleReporter.SampleObserver.ObserveSample(sample)
}

func (r *ReporterWithDimensions) ReportDuration(name string, duration time.Duration, dimensions ...DimensionSet) {
	if len(dimensions) == 0 {
		r.SampleReporter.ReportDurationSample(name, duration)
	}

	builder := r.builderWithDimensions(dimensions)
	// create sample
	sample, err := builder.BuildDurationSample(name, duration)
	if err != nil {
		log.Error(err)
		return
	}

	// observe sample
	r.SampleReporter.SampleObserver.ObserveSample(sample)
}

func (r *ReporterWithDimensions) builderWithDimensions(dimensions []DimensionSet) telemetry.SampleBuilder {
	// copy builder dimensions
	combinedDimensions := make(map[string]string)
	for key, value := range r.SampleReporter.SampleBuilder.Dimensions {
		combinedDimensions[key] = value
	}

	// create new builder
	for _, dimension := range dimensions {
		for name, value := range dimension {
			combinedDimensions[name] = value
		}
	}

	builder := r.SampleReporter.SampleBuilder

	return telemetry.SampleBuilder{
		ProcessIdentifier:             builder.ProcessIdentifier,
		OperationName:                 builder.OperationName,
		DependencyProcessIdentifier:   builder.DependencyProcessIdentifier,
		DependencyOperationName:       builder.DependencyOperationName,
		Dimensions:                    combinedDimensions,
		Timestamp:                     builder.Timestamp,
		EnableProcessAddressDimension: builder.EnableProcessAddressDimension,
	}
}
